document.addEventListener('DOMContentLoaded', function() {
    initNavigation();
    initModals();
    initGalleryFilter();
    initTestimonialSlider();
    initFormHandlers();
    initBackToTop();
    initStickyHeader();
    initSmoothScroll();
    initNotifications();
    initDynamicContent();
    initMobileOptimizations();
    initLazyLoading();
    initAnimations();
    initMapInteraction();
    initSearchFunctionality();
    initBookingSystem();
    initDestinationFilter();
    initPackageComparison();
    initFavorites();
    initShareButtons();
    initRatingSystem();
    initAccordions();
    initCounters();
    initValidation();
    initLanguageSelector();
    initCurrencyConverter();
    initWeatherWidget();
    initCountdownTimers();
    initTips();
    initPrint();
    initThemeSwitcher();
    initAccessibility();
});

function initNavigation() {
    const hamburger = document.querySelector('.hamburger');
    const navMenu = document.getElementById('nav-menu');
    const navLinks = document.querySelectorAll('#nav-menu a');
    const header = document.querySelector('header');
    
    if (hamburger) {
        hamburger.addEventListener('click', function() {
            toggleMenu();
        });
    }
    
    if (navLinks) {
        navLinks.forEach(link => {
            link.addEventListener('click', function() {
                if (window.innerWidth <= 768 && navMenu.classList.contains('active')) {
                    toggleMenu();
                }
            });
        });
    }
    
    function toggleMenu() {
        if (navMenu) {
            navMenu.classList.toggle('active');
        }
        
        if (hamburger) {
            hamburger.classList.toggle('active');
        }
        
        if (document.body) {
            document.body.classList.toggle('menu-open');
        }
    }
    
    document.addEventListener('click', function(e) {
        if (navMenu && hamburger && navMenu.classList.contains('active')) {
            if (!navMenu.contains(e.target) && !hamburger.contains(e.target)) {
                toggleMenu();
            }
        }
    });
    
    window.addEventListener('resize', function() {
        if (window.innerWidth > 768 && navMenu && navMenu.classList.contains('active')) {
            toggleMenu();
        }
    });
}

function initModals() {
    const modalTriggers = document.querySelectorAll('[onclick*="openModal"]');
    const modalCloseButtons = document.querySelectorAll('.close-modal');
    const modals = document.querySelectorAll('.modal');
    
    modalTriggers.forEach(trigger => {
        trigger.addEventListener('click', function(e) {
            e.preventDefault();
            const modalId = this.getAttribute('onclick').split('\'')[1];
            openModal(modalId);
        });
    });
    
    modalCloseButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const modalId = this.closest('.modal').id;
            closeModal(modalId);
        });
    });
    
    window.addEventListener('click', function(e) {
        modals.forEach(modal => {
            if (e.target === modal) {
                closeModal(modal.id);
            }
        });
    });
    
    window.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            const activeModal = document.querySelector('.modal.active');
            if (activeModal) {
                closeModal(activeModal.id);
            }
        }
    });
}

function openModal(modalId) {
    const activeModals = document.querySelectorAll('.modal.active');
    const targetModal = document.getElementById(modalId);
    
    if (!targetModal) return;
    
    activeModals.forEach(modal => {
        modal.classList.remove('active');
    });
    
    targetModal.classList.add('active');
    document.body.classList.add('modal-open');
    
    const modalContent = targetModal.querySelector('.modal-content');
    if (modalContent) {
        modalContent.scrollTop = 0;
    }
    
    if (modalId === 'modal-gallery') {
        initGallerySlider();
    }
    
    const modalForm = targetModal.querySelector('form');
    if (modalForm) {
        modalForm.reset();
    }
    
    if (targetModal.classList.contains('package-modal') || targetModal.classList.contains('destination-modal')) {
        targetModal.classList.add('zoom-animation');
        setTimeout(() => {
            targetModal.classList.remove('zoom-animation');
        }, 500);
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    
    if (!modal) return;
    
    modal.classList.add('fade-out');
    
    setTimeout(() => {
        modal.classList.remove('active');
        modal.classList.remove('fade-out');
        
        const activeModals = document.querySelectorAll('.modal.active');
        if (activeModals.length === 0) {
            document.body.classList.remove('modal-open');
        }
    }, 300);
}

function changeMainImage(thumbnailImg) {
    if (!thumbnailImg) return;
    
    const mainImageContainer = thumbnailImg.closest('.modal-image-gallery');
    if (!mainImageContainer) return;
    
    const mainImage = mainImageContainer.querySelector('.main-image');
    const thumbnails = mainImageContainer.querySelectorAll('.thumbnail-images img');
    
    if (mainImage) {
        mainImage.classList.add('image-transition');
        
        setTimeout(() => {
            mainImage.src = thumbnailImg.src;
            mainImage.alt = thumbnailImg.alt;
            mainImage.classList.remove('image-transition');
        }, 300);
    }
    
    if (thumbnails) {
        thumbnails.forEach(thumb => {
            thumb.classList.remove('active');
        });
        thumbnailImg.classList.add('active');
    }
    
    if (mainImageContainer.classList.contains('zoom-enabled')) {
        initImageZoom();
    }
}

function initGalleryFilter() {
    const filterButtons = document.querySelectorAll('.filter-btn');
    const galleryItems = document.querySelectorAll('.gallery-item');
    
    if (!filterButtons.length || !galleryItems.length) return;
    
    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            const filter = this.getAttribute('data-filter');
            
            filterButtons.forEach(btn => {
                btn.classList.remove('active');
            });
            
            this.classList.add('active');
            
            galleryItems.forEach(item => {
                if (filter === 'all') {
                    item.style.display = 'block';
                    setTimeout(() => {
                        item.classList.add('fade-in');
                    }, 50);
                } else if (item.getAttribute('data-category') === filter) {
                    item.style.display = 'block';
                    setTimeout(() => {
                        item.classList.add('fade-in');
                    }, 50);
                } else {
                    item.classList.remove('fade-in');
                    setTimeout(() => {
                        item.style.display = 'none';
                    }, 300);
                }
            });
            
            reorganizeGallery();
        });
    });
    
    function reorganizeGallery() {
        const visibleItems = Array.from(galleryItems).filter(item => item.style.display !== 'none');
        const container = document.querySelector('.gallery-container');
        
        if (!container) return;
        
        setTimeout(() => {
            if (window.innerWidth > 768) {
                const itemsPerRow = window.innerWidth > 1024 ? 4 : 3;
                const rows = Math.ceil(visibleItems.length / itemsPerRow);
                
                for (let i = 0; i < rows; i++) {
                    const rowItems = visibleItems.slice(i * itemsPerRow, (i + 1) * itemsPerRow);
                    rowItems.forEach((item, index) => {
                        const delay = (i * itemsPerRow + index) * 50;
                        setTimeout(() => {
                            item.classList.add('appear');
                        }, delay);
                    });
                }
            } else {
                visibleItems.forEach((item, index) => {
                    const delay = index * 50;
                    setTimeout(() => {
                        item.classList.add('appear');
                    }, delay);
                });
            }
        }, 300);
    }
}

function initTestimonialSlider() {
    const testimonialSlider = document.querySelector('.testimonial-slider');
    const testimonialItems = document.querySelectorAll('.testimonial-item');
    const dots = document.querySelectorAll('.testimonial-dots .dot');
    const prevButton = document.querySelector('.testimonial-controls .prev-btn');
    const nextButton = document.querySelector('.testimonial-controls .next-btn');
    
    if (!testimonialSlider || !testimonialItems.length) return;
    
    let currentSlide = 0;
    let touchStartX = 0;
    let touchEndX = 0;
    let autoSlideTimer;
    
    function showSlide(n) {
        const slidesCount = testimonialItems.length;
        
        if (n >= slidesCount) {
            currentSlide = 0;
        } else if (n < 0) {
            currentSlide = slidesCount - 1;
        } else {
            currentSlide = n;
        }
        
        testimonialItems.forEach((item, index) => {
            if (index === currentSlide) {
                item.style.display = 'block';
                item.classList.add('fade-in');
            } else {
                item.classList.remove('fade-in');
                setTimeout(() => {
                    item.style.display = 'none';
                }, 400);
            }
        });
        
        dots.forEach((dot, index) => {
            dot.classList.remove('active');
            if (index === currentSlide) {
                dot.classList.add('active');
            }
        });
    }
    
    function startAutoSlide() {
        clearInterval(autoSlideTimer);
        autoSlideTimer = setInterval(() => {
            showSlide(currentSlide + 1);
        }, 5000);
    }
    
    function stopAutoSlide() {
        clearInterval(autoSlideTimer);
    }
    
    if (prevButton) {
        prevButton.addEventListener('click', function() {
            showSlide(currentSlide - 1);
            stopAutoSlide();
            startAutoSlide();
        });
    }
    
    if (nextButton) {
        nextButton.addEventListener('click', function() {
            showSlide(currentSlide + 1);
            stopAutoSlide();
            startAutoSlide();
        });
    }
    
    dots.forEach((dot, index) => {
        dot.addEventListener('click', function() {
            showSlide(index);
            stopAutoSlide();
            startAutoSlide();
        });
    });
    
    testimonialSlider.addEventListener('touchstart', function(e) {
        touchStartX = e.changedTouches[0].screenX;
    });
    
    testimonialSlider.addEventListener('touchend', function(e) {
        touchEndX = e.changedTouches[0].screenX;
        handleSwipe();
    });
    
    function handleSwipe() {
        if (touchEndX < touchStartX - 50) {
            showSlide(currentSlide + 1);
            stopAutoSlide();
            startAutoSlide();
        } else if (touchEndX > touchStartX + 50) {
            showSlide(currentSlide - 1);
            stopAutoSlide();
            startAutoSlide();
        }
    }
    
    showSlide(0);
    startAutoSlide();
}

function initFormHandlers() {
    const forms = document.querySelectorAll('form');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            if (!validateForm(this)) {
                return false;
            }
            
            const formData = new FormData(this);
            const formEntries = {};
            
            for (let [key, value] of formData.entries()) {
                formEntries[key] = value;
            }
            
            const formId = this.id;
            
            if (formId === 'contactForm' || formId === 'contactModalForm') {
                handleContactForm(formEntries, this);
            } else if (formId === 'bookingForm') {
                handleBookingForm(formEntries, this);
            } else if (formId === 'newsletterForm' || this.classList.contains('newsletter-form')) {
                handleNewsletterForm(formEntries, this);
            } else {
                handleGenericForm(formEntries, this);
            }
        });
    });
    
    const healthYesRadio = document.getElementById('healthYes');
    const healthNoRadio = document.getElementById('healthNo');
    const healthDetailsContainer = document.getElementById('healthDetailsContainer');
    
    if (healthYesRadio && healthNoRadio && healthDetailsContainer) {
        healthYesRadio.addEventListener('change', function() {
            healthDetailsContainer.style.display = this.checked ? 'block' : 'none';
        });
        
        healthNoRadio.addEventListener('change', function() {
            healthDetailsContainer.style.display = 'none';
        });
    }
    
    function validateForm(form) {
        const requiredFields = form.querySelectorAll('[required]');
        let isValid = true;
        
        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                markInvalid(field, 'Bidang ini wajib diisi');
                isValid = false;
            } else {
                clearInvalid(field);
                
                if (field.type === 'email' && !validateEmail(field.value)) {
                    markInvalid(field, 'Email tidak valid');
                    isValid = false;
                } else if (field.type === 'tel' && !validatePhone(field.value)) {
                    markInvalid(field, 'Nomor telepon tidak valid');
                    isValid = false;
                }
            }
        });
        
        return isValid;
    }
    
    function markInvalid(field, message) {
        field.classList.add('invalid');
        
        let errorMsg = field.parentElement.querySelector('.error-message');
        
        if (!errorMsg) {
            errorMsg = document.createElement('span');
            errorMsg.className = 'error-message';
            field.parentElement.appendChild(errorMsg);
        }
        
        errorMsg.textContent = message;
    }
    
    function clearInvalid(field) {
        field.classList.remove('invalid');
        
        const errorMsg = field.parentElement.querySelector('.error-message');
        if (errorMsg) {
            errorMsg.remove();
        }
    }
    
    function validateEmail(email) {
        const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return regex.test(email);
    }
    
    function validatePhone(phone) {
        const regex = /^[+]?[\d\s-]{8,15}$/;
        return regex.test(phone);
    }
    
    function handleContactForm(data, form) {
        showFormLoader(form);
        
        setTimeout(() => {
            hideFormLoader(form);
            
            closeModal('modal-contact');
            form.reset();
            
            showNotification('Pesan Terkirim', 'Terima kasih! Pesan Anda telah kami terima. Tim kami akan menghubungi Anda secepatnya.', 'success');
        }, 1500);
    }
    
    function handleBookingForm(data, form) {
        showFormLoader(form);
        
        setTimeout(() => {
            hideFormLoader(form);
            
            closeModal('modal-booking');
            form.reset();
            
            openModal('modal-booking-success');
        }, 2000);
    }
    
    function handleNewsletterForm(data, form) {
        showFormLoader(form);
        
        setTimeout(() => {
            hideFormLoader(form);
            
            form.reset();
            
            if (form.closest('.modal')) {
                closeModal(form.closest('.modal').id);
            }
            
            showNotification('Berlangganan Newsletter', 'Terima kasih telah berlangganan newsletter kami! Anda akan menerima penawaran eksklusif dan informasi terbaru.', 'success');
        }, 1000);
    }
    
    function handleGenericForm(data, form) {
        showFormLoader(form);
        
        setTimeout(() => {
            hideFormLoader(form);
            form.reset();
            
            showNotification('Formulir Terkirim', 'Terima kasih! Data Anda telah kami terima.', 'success');
        }, 1500);
    }
    
    function showFormLoader(form) {
        const submitButton = form.querySelector('[type="submit"]');
        
        if (submitButton) {
            submitButton.disabled = true;
            submitButton.classList.add('loading');
            
            const originalText = submitButton.textContent;
            submitButton.setAttribute('data-original-text', originalText);
            submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Memproses...';
        }
    }
    
    function hideFormLoader(form) {
        const submitButton = form.querySelector('[type="submit"]');
        
        if (submitButton) {
            submitButton.disabled = false;
            submitButton.classList.remove('loading');
            
            const originalText = submitButton.getAttribute('data-original-text');
            submitButton.textContent = originalText;
        }
    }
}

function initBackToTop() {
    const backToTopButton = document.getElementById('backToTop');
    
    if (!backToTopButton) return;
    
    window.addEventListener('scroll', function() {
        if (window.pageYOffset > 300) {
            backToTopButton.classList.add('show');
        } else {
            backToTopButton.classList.remove('show');
        }
    });
    
    backToTopButton.addEventListener('click', function() {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });
}

function initStickyHeader() {
    const header = document.querySelector('header');
    
    if (!header) return;
    
    const headerHeight = header.offsetHeight;
    const hero = document.querySelector('.hero-section');
    
    if (hero) {
        hero.style.paddingTop = `${headerHeight}px`;
    }
    
    window.addEventListener('scroll', function() {
        if (window.pageYOffset > 50) {
            header.classList.add('sticky');
        } else {
            header.classList.remove('sticky');
        }
    });
}

function initSmoothScroll() {
    const scrollLinks = document.querySelectorAll('a[href^="#"]');
    
    scrollLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            const href = this.getAttribute('href');
            
            if (href === '#') return;
            
            e.preventDefault();
            
            const targetElement = document.querySelector(href);
            
            if (!targetElement) return;
            
            const headerHeight = document.querySelector('header').offsetHeight;
            const targetPosition = targetElement.getBoundingClientRect().top + window.pageYOffset - headerHeight;
            
            window.scrollTo({
                top: targetPosition,
                behavior: 'smooth'
            });
        });
    });
}

function initNotifications() {
    const notificationPopup = document.getElementById('notification-popup');
    const closeNotificationButton = document.querySelector('.close-notification');
    
    if (notificationPopup) {
        setTimeout(() => {
            notificationPopup.classList.add('show');
        }, 5000);
        
        if (closeNotificationButton) {
            closeNotificationButton.addEventListener('click', function() {
                closeNotification();
            });
        }
    }
    
    function closeNotification() {
        if (notificationPopup) {
            notificationPopup.classList.remove('show');
            
            localStorage.setItem('notification_closed', Date.now());
        }
    }
    
    window.showNotification = function(title, message, type = 'info') {
        const notificationContainer = document.querySelector('.custom-notifications');
        
        if (!notificationContainer) {
            const container = document.createElement('div');
            container.className = 'custom-notifications';
            document.body.appendChild(container);
        }
        
        const notification = document.createElement('div');
        notification.className = `custom-notification ${type}`;
        
        notification.innerHTML = `
            <div class="notification-header">
                <h4>${title}</h4>
                <button class="close-btn"><i class="fas fa-times"></i></button>
            </div>
            <div class="notification-body">
                <p>${message}</p>
            </div>
        `;
        
        document.querySelector('.custom-notifications').appendChild(notification);
        
        setTimeout(() => {
            notification.classList.add('show');
        }, 10);
        
        const closeBtn = notification.querySelector('.close-btn');
        
        if (closeBtn) {
            closeBtn.addEventListener('click', function() {
                removeNotification(notification);
            });
        }
        
        setTimeout(() => {
            removeNotification(notification);
        }, 5000);
    }
    
    function removeNotification(notification) {
        notification.classList.remove('show');
        
        setTimeout(() => {
            notification.remove();
        }, 300);
    }
}

function initDynamicContent() {
    generateRandomContent();
    addRandomPromoTag();
    updateDates();
    populateDestinations();
    
    function generateRandomContent() {
        const days = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
        const today = new Date();
        const dayName = days[today.getDay()];
        
        const quotes = [
            "Hidup adalah petualangan berani atau bukan sama sekali.",
            "Perjalanan panjang dimulai dengan langkah pertama.",
            "Temukan dirimu di tengah alam liar.",
            "Gunung tidak meminta kita untuk datang, kita yang memilih untuk mengunjunginya.",
            "Bumi tidak mewarisi kita, kita meminjamnya dari anak cucu kita.",
            "Perjalanan sejati bukan tentang menemukan tempat baru, tetapi melihat dengan mata baru.",
            "Hidup seperti mendaki gunung, ada kalanya sulit, tapi pemandangannya luar biasa."
        ];
        
        const randomQuote = quotes[today.getDay()];
        
        const quoteContainer = document.querySelector('.hero-content p');
        if (quoteContainer) {
            quoteContainer.innerHTML = `<em>"${randomQuote}"</em> - Quote hari ${dayName}`;
        }
        
        const promos = [
            "Diskon 10% untuk pemesanan di hari Minggu",
            "Bonus welcome drink untuk pemesanan di hari Senin",
            "Paket keluarga hemat di hari Selasa",
            "Diskon 15% untuk grup di hari Rabu", 
            "Gratis dokumentasi untuk pemesanan di hari Kamis",
            "Special weekend package di hari Jumat",
            "Extra night stay di hari Sabtu"
        ];
        
        const randomPromo = promos[today.getDay()];
        
        const notificationContent = document.querySelector('.notification-content p');
        if (notificationContent) {
            notificationContent.textContent = randomPromo;
        }
    }
    
    function addRandomPromoTag() {
        const promoTags = ['HOT DEAL!', 'BEST SELLER!', 'LIMITED!', 'NEW!'];
        const packages = document.querySelectorAll('.package-card');
        
        packages.forEach(package => {
            const randomIndex = Math.floor(Math.random() * promoTags.length);
            const randomTag = promoTags[randomIndex];
            const existingTag = package.querySelector('.package-tag');
            
            if (existingTag) {
                existingTag.textContent = randomTag;
                
                if (randomTag === 'HOT DEAL!') {
                    existingTag.classList.add('hot-deal');
                } else if (randomTag === 'BEST SELLER!') {
                    existingTag.classList.add('best-seller');
                } else if (randomTag === 'LIMITED!') {
                    existingTag.classList.add('limited');
                } else if (randomTag === 'NEW!') {
                    existingTag.classList.add('new-tag');
                }
            }
        });
    }
    
    function updateDates() {
        const currentYear = new Date().getFullYear();
        const copyrightElement = document.querySelector('.copyright p');
        
        if (copyrightElement) {
            copyrightElement.textContent = `© ${currentYear} Camp IDN. All Rights Reserved.`;
        }
        
        const dateInputs = document.querySelectorAll('input[type="date"]');
        
        dateInputs.forEach(input => {
            const today = new Date();
            const minDate = new Date();
            const maxDate = new Date();
            
            minDate.setDate(today.getDate() + 7);
            maxDate.setFullYear(today.getFullYear() + 1);
            
            input.min = minDate.toISOString().split('T')[0];
            input.max = maxDate.toISOString().split('T')[0];
        });
    }
    
    function populateDestinations() {
        const destinations = [
            { name: "Gunung Rinjani", location: "Lombok, NTB", type: "gunung", height: "3.726 mdpl" },
            { name: "Raja Ampat", location: "Papua Barat", type: "laut", feature: "1.500+ spesies ikan" },
            { name: "Taman Nasional Bromo Tengger Semeru", location: "Jawa Timur", type: "gunung", height: "2.329 mdpl" },
            { name: "Taman Nasional Komodo", location: "Nusa Tenggara Timur", type: "laut", feature: "Laut Flores" },
            { name: "Danau Toba", location: "Sumatera Utara", type: "danau", size: "1.145 km²" },
            { name: "Wakatobi", location: "Sulawesi Tenggara", type: "laut", feature: "1.390.000 ha" },
            { name: "Gunung Kerinci", location: "Jambi", type: "gunung", height: "3.805 mdpl" },
            { name: "Pulau Derawan", location: "Kalimantan Timur", type: "laut", feature: "Penyu Hijau" },
            { name: "Taman Nasional Gunung Leuser", location: "Aceh", type: "hutan", feature: "Orangutan Sumatera" },
            { name: "Danau Kelimutu", location: "Flores, NTT", type: "danau", feature: "Tiga Warna Air" }
        ];
        
        const destinationSelect = document.getElementById('destination');
        
        if (destinationSelect) {
            destinations.forEach(dest => {
                const option = document.createElement('option');
                option.value = dest.name.toLowerCase().replace(/\s+/g, '-');
                option.textContent = `${dest.name}, ${dest.location}`;
                
                destinationSelect.appendChild(option);
            });
        }
    }
}

function initMobileOptimizations() {
    const isMobile = window.innerWidth <= 768;
    
    if (isMobile) {
        optimizeImagesForMobile();
        simplifyLayoutForMobile();
        enhanceTouchInteractions();
    }
    
    window.addEventListener('resize', function() {
        if (window.innerWidth <= 768 && !isMobile) {
            optimizeImagesForMobile();
            simplifyLayoutForMobile();
            enhanceTouchInteractions();
        } else if (window.innerWidth > 768 && isMobile) {
            restoreDesktopExperience();
        }
    });
    
    function optimizeImagesForMobile() {
        const largeImages = document.querySelectorAll('img:not(.thumbnail-images img)');
        
        largeImages.forEach(img => {
            const originalSrc = img.src;
            
            if (originalSrc.includes('/api/placeholder/')) {
                const parts = originalSrc.split('/');
                const width = parseInt(parts[parts.length - 2]);
                const height = parseInt(parts[parts.length - 1]);
                
                if (width > 400) {
                    const newWidth = Math.min(width, window.innerWidth - 40);
                    const newHeight = Math.floor((height * newWidth) / width);
                    
                    parts[parts.length - 2] = newWidth.toString();
                    parts[parts.length - 1] = newHeight.toString();
                    
                    img.src = parts.join('/');
                }
            }
        });
    }
    
    function simplifyLayoutForMobile() {
        const destinationCards = document.querySelectorAll('.destination-card');
        
        destinationCards.forEach(card => {
            const metaInfo = card.querySelector('.destination-meta');
            
            if (metaInfo) {
                const spans = metaInfo.querySelectorAll('span');
                
                if (spans.length > 1) {
                    spans[0].style.display = 'block';
                    spans[1].style.display = 'block';
                }
            }
        });
        
        const packageFeatures = document.querySelectorAll('.package-features li');
        
        if (packageFeatures.length > 4) {
            for (let i = 4; i < packageFeatures.length; i++) {
                packageFeatures[i].classList.add('hidden-mobile');
            }
            
            const showMoreButton = document.createElement('button');
            showMoreButton.className = 'show-more-features';
            showMoreButton.textContent = 'Lihat Semua Fitur';
            
            if (packageFeatures[0].parentElement) {
                packageFeatures[0].parentElement.after(showMoreButton);
                
                showMoreButton.addEventListener('click', function() {
                    packageFeatures.forEach(feature => {
                        feature.classList.remove('hidden-mobile');
                    });
                    this.style.display = 'none';
                });
            }
        }
        
        const galleryFilter = document.querySelector('.gallery-filter');
        
        if (galleryFilter) {
            const filterButtons = galleryFilter.querySelectorAll('.filter-btn');
            const filterContainer = document.createElement('div');
            filterContainer.className = 'mobile-filter-dropdown';
            
            const filterSelected = document.createElement('div');
            filterSelected.className = 'filter-selected';
            filterSelected.textContent = 'Semua';
            
            const filterOptions = document.createElement('div');
            filterOptions.className = 'filter-options';
            
            filterOptions.style.display = 'none';
            
            filterButtons.forEach(button => {
                const option = document.createElement('div');
                option.className = 'filter-option';
                option.setAttribute('data-filter', button.getAttribute('data-filter'));
                option.textContent = button.textContent;
                
                option.addEventListener('click', function() {
                    filterSelected.textContent = this.textContent;
                    filterOptions.style.display = 'none';
                    
                    const filter = this.getAttribute('data-filter');
                    
                    filterButtons.forEach(btn => {
                        if (btn.getAttribute('data-filter') === filter) {
                            btn.click();
                        }
                    });
                });
                
                filterOptions.appendChild(option);
            });
            
            filterSelected.addEventListener('click', function() {
                filterOptions.style.display = filterOptions.style.display === 'none' ? 'block' : 'none';
            });
            
            filterContainer.appendChild(filterSelected);
            filterContainer.appendChild(filterOptions);
            
            galleryFilter.style.display = 'none';
            galleryFilter.parentElement.insertBefore(filterContainer, galleryFilter);
            
            document.addEventListener('click', function(e) {
                if (!filterContainer.contains(e.target)) {
                    filterOptions.style.display = 'none';
                }
            });
        }
    }
    
    function enhanceTouchInteractions() {
        const destinationCards = document.querySelectorAll('.destination-card');
        
        destinationCards.forEach(card => {
            const viewButton = card.querySelector('.btn-view-details');
            
            if (viewButton) {
                card.addEventListener('click', function(e) {
                    if (e.target !== viewButton && !viewButton.contains(e.target)) {
                        viewButton.click();
                    }
                });
            }
        });
        
        const galleryItems = document.querySelectorAll('.gallery-item');
        
        galleryItems.forEach(item => {
            let touchStartY = 0;
            let touchEndY = 0;
            
            item.addEventListener('touchstart', function(e) {
                touchStartY = e.changedTouches[0].screenY;
            });
            
            item.addEventListener('touchend', function(e) {
                touchEndY = e.changedTouches[0].screenY;
                
                if (Math.abs(touchEndY - touchStartY) < 30) {
                    const modalId = 'modal-gallery';
                    const galleryModal = document.getElementById(modalId);
                    
                    if (!galleryModal) {
                        createGalleryModal();
                    }
                    
                    openModal(modalId);
                    
                    const mainImage = document.querySelector('#modal-gallery .main-image');
                    const itemImage = item.querySelector('img');
                    
                    if (mainImage && itemImage) {
                        mainImage.src = itemImage.src;
                        mainImage.alt = itemImage.alt;
                    }
                    
                    const galleryTitle = document.querySelector('#modal-gallery .modal-header h2');
                    const itemTitle = item.querySelector('.gallery-overlay h3');
                    
                    if (galleryTitle && itemTitle) {
                        galleryTitle.textContent = itemTitle.textContent;
                    }
                }
            });
        });
        
        function createGalleryModal() {
            const modal = document.createElement('div');
            modal.className = 'modal';
            modal.id = 'modal-gallery';
            
            modal.innerHTML = `
                <div class="modal-content">
                    <span class="close-modal" onclick="closeModal('modal-gallery')">&times;</span>
                    <div class="modal-header">
                        <h2>Galeri</h2>
                    </div>
                    <div class="modal-body">
                        <div class="modal-image-gallery">
                            <img src="/api/placeholder/800/500" alt="Gallery Image" class="main-image">
                            <div class="gallery-controls">
                                <button class="gallery-prev"><i class="fas fa-chevron-left"></i></button>
                                <button class="gallery-next"><i class="fas fa-chevron-right"></i></button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            document.body.appendChild(modal);
            
            const prevButton = modal.querySelector('.gallery-prev');
            const nextButton = modal.querySelector('.gallery-next');
            
            if (prevButton && nextButton) {
                prevButton.addEventListener('click', function() {
                    navigateGallery(-1);
                });
                
                nextButton.addEventListener('click', function() {
                    navigateGallery(1);
                });
            }
        }
        
        function navigateGallery(direction) {
            const galleryItems = document.querySelectorAll('.gallery-item');
            const mainImage = document.querySelector('#modal-gallery .main-image');
            const galleryTitle = document.querySelector('#modal-gallery .modal-header h2');
            
            if (!mainImage || !galleryTitle || !galleryItems.length) return;
            
            let currentIndex = -1;
            
            for (let i = 0; i < galleryItems.length; i++) {
                const itemImage = galleryItems[i].querySelector('img');
                
                if (itemImage && itemImage.src === mainImage.src) {
                    currentIndex = i;
                    break;
                }
            }
            
            if (currentIndex === -1) return;
            
            let newIndex = currentIndex + direction;
            
            if (newIndex < 0) {
                newIndex = galleryItems.length - 1;
            } else if (newIndex >= galleryItems.length) {
                newIndex = 0;
            }
            
            const newItemImage = galleryItems[newIndex].querySelector('img');
            const newItemTitle = galleryItems[newIndex].querySelector('.gallery-overlay h3');
            
            if (newItemImage) {
                mainImage.src = newItemImage.src;
                mainImage.alt = newItemImage.alt;
            }
            
            if (newItemTitle) {
                galleryTitle.textContent = newItemTitle.textContent;
            }
        }
    }
    
    function restoreDesktopExperience() {
        const showMoreButton = document.querySelector('.show-more-features');
        
        if (showMoreButton) {
            showMoreButton.remove();
        }
        
        const hiddenFeatures = document.querySelectorAll('.package-features li.hidden-mobile');
        
        hiddenFeatures.forEach(feature => {
            feature.classList.remove('hidden-mobile');
        });
        
        const mobileFilterDropdown = document.querySelector('.mobile-filter-dropdown');
        const galleryFilter = document.querySelector('.gallery-filter');
        
        if (mobileFilterDropdown && galleryFilter) {
            mobileFilterDropdown.remove();
            galleryFilter.style.display = 'flex';
        }
    }
}

function initLazyLoading() {
    const lazyImages = document.querySelectorAll('img[data-src]');
    
    if (!lazyImages.length) return;
    
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.getAttribute('data-src');
                img.removeAttribute('data-src');
                img.classList.add('loaded');
                observer.unobserve(img);
            }
        });
    });
    
    lazyImages.forEach(img => {
        imageObserver.observe(img);
    });
}

function initAnimations() {
    const animElements = document.querySelectorAll('.animate-on-scroll');
    
    if (!animElements.length) return;
    
    const animObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animated');
            }
        });
    }, { threshold: 0.1 });
    
    animElements.forEach(element => {
        animObserver.observe(element);
    });
    
    const sections = document.querySelectorAll('section');
    
    const sectionObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('section-visible');
                
                const staggerElems = entry.target.querySelectorAll('.stagger-anim');
                
                staggerElems.forEach((elem, index) => {
                    setTimeout(() => {
                        elem.classList.add('stagger-visible');
                    }, 100 * index);
                });
            }
        });
    }, { threshold: 0.1 });
    
    sections.forEach(section => {
        sectionObserver.observe(section);
    });
}

function initMapInteraction() {
    const mapContainer = document.querySelector('.map');
    
    if (!mapContainer) return;
    
    mapContainer.addEventListener('click', function() {
        this.classList.toggle('expanded');
    });
}

function initSearchFunctionality() {
    const searchIcon = document.createElement('div');
    searchIcon.className = 'search-icon';
    searchIcon.innerHTML = '<i class="fas fa-search"></i>';
    
    const header = document.querySelector('header .header-container');
    
    if (header) {
        header.appendChild(searchIcon);
    }
    
    const searchModal = document.createElement('div');
    searchModal.className = 'modal search-modal';
    searchModal.id = 'modal-search';
    
    searchModal.innerHTML = `
        <div class="modal-content">
            <span class="close-modal" onclick="closeModal('modal-search')">&times;</span>
            <div class="search-container">
                <form class="search-form">
                    <input type="text" placeholder="Cari destinasi, paket, atau artikel..." autofocus>
                    <button type="submit"><i class="fas fa-search"></i></button>
                </form>
                <div class="popular-searches">
                    <h3>Pencarian Populer</h3>
                    <div class="popular-tags">
                        <a href="#" class="tag">Gunung Rinjani</a>
                        <a href="#" class="tag">Raja Ampat</a>
                        <a href="#" class="tag">Komodo</a>
                        <a href="#" class="tag">Diving</a>
                        <a href="#" class="tag">Pendakian</a>
                        <a href="#" class="tag">Paket Keluarga</a>
                    </div>
                </div>
                <div class="search-results"></div>
            </div>
        </div>
    `;
    
    document.body.appendChild(searchModal);
    
    searchIcon.addEventListener('click', function() {
        openModal('modal-search');
        
        setTimeout(() => {
            const searchInput = document.querySelector('.search-form input');
            if (searchInput) {
                searchInput.focus();
            }
        }, 100);
    });
    
    const searchForm = document.querySelector('.search-form');
    
    if (searchForm) {
        searchForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const searchTerm = this.querySelector('input').value.trim();
            
            if (!searchTerm) return;
            
            performSearch(searchTerm);
        });
    }
    
    function performSearch(term) {
        const searchResults = document.querySelector('.search-results');
        
        if (!searchResults) return;
        
        searchResults.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i> Mencari...</div>';
        
        setTimeout(() => {
            const results = [];
            
            const destinations = document.querySelectorAll('.destination-card');
            const packages = document.querySelectorAll('.package-card');
            const blogs = document.querySelectorAll('.blog-post');
            
            destinations.forEach(dest => {
                const title = dest.querySelector('h3');
                
                if (title && title.textContent.toLowerCase().includes(term.toLowerCase())) {
                    results.push({
                        type: 'destination',
                        title: title.textContent,
                        image: dest.querySelector('img').src,
                        link: '#destinasi'
                    });
                }
            });
            
            packages.forEach(pack => {
                const title = pack.querySelector('h3');
                
                if (title && title.textContent.toLowerCase().includes(term.toLowerCase())) {
                    results.push({
                        type: 'package',
                        title: title.textContent,
                        image: pack.querySelector('img').src,
                        link: '#paket'
                    });
                }
            });
            
            blogs.forEach(blog => {
                const title = blog.querySelector('h3');
                
                if (title && title.textContent.toLowerCase().includes(term.toLowerCase())) {
                    results.push({
                        type: 'blog',
                        title: title.textContent,
                        image: blog.querySelector('img').src,
                        link: '#blog'
                    });
                }
            });
            
            if (results.length) {
                let resultsHTML = '<h3>Hasil Pencarian</h3><div class="results-grid">';
                
                results.forEach(result => {
                    resultsHTML += `
                        <div class="result-item ${result.type}">
                            <img src="${result.image}" alt="${result.title}">
                            <div class="result-info">
                                <span class="result-type">${result.type === 'destination' ? 'Destinasi' : result.type === 'package' ? 'Paket' : 'Artikel'}</span>
                                <h4>${result.title}</h4>
                                <a href="${result.link}" onclick="closeModal('modal-search')">Lihat Detail</a>
                            </div>
                        </div>
                    `;
                });
                
                resultsHTML += '</div>';
                searchResults.innerHTML = resultsHTML;
            } else {
                searchResults.innerHTML = `
                    <div class="no-results">
                        <i class="fas fa-search"></i>
                        <h3>Tidak Ditemukan</h3>
                        <p>Maaf, tidak ada hasil untuk pencarian "${term}". Coba kata kunci lain.</p>
                    </div>
                `;
            }
        }, 1000);
    }
}

function initBookingSystem() {
    const packageSelects = document.querySelectorAll('select[name="package"]');
    const dateInputs = document.querySelectorAll('input[name="departureDate"]');
    const participantsInputs = document.querySelectorAll('input[name="participants"]');
    
    if (!packageSelects.length) return;
    
    const packages = {
        'rinjani-basic': {
            name: 'Rinjani Basic (3D2N)',
            price: 2500000,
            minParticipants: 2,
            maxParticipants: 10,
            availableDates: [7, 14, 21, 28] 
        },
        'rinjani-complete': {
            name: 'Rinjani Complete (4D3N)',
            price: 3800000,
            minParticipants: 2,
            maxParticipants: 8,
            availableDates: [5, 12, 19, 26]
        },
        'puncak-nusantara': {
            name: 'Ekspedisi Puncak Nusantara (5D4N)',
            price: 5500000,
            minParticipants: 4,
            maxParticipants: 8,
            availableDates: [3, 10, 17, 24]
        },
        'pesona-bawah-laut': {
            name: 'Pesona Bawah Laut (4D3N)',
            price: 4800000,
            minParticipants: 2,
            maxParticipants: 10,
            availableDates: [2, 9, 16, 23, 30]
        },
        'eksplorasi-budaya': {
            name: 'Eksplorasi Budaya Nusantara (7D6N)',
            price: 6200000,
            minParticipants: 4,
            maxParticipants: 12,
            availableDates: [1, 15, 29]
        },
        'adrenalin-sungai': {
            name: 'Adrenalin Sungai Liar (3D2N)',
            price: 3200000,
            minParticipants: 4,
            maxParticipants: 8,
            availableDates: [4, 11, 18, 25]
        }
    };
    
    packageSelects.forEach(select => {
        select.addEventListener('change', function() {
            const packageValue = this.value;
            const form = this.closest('form');
            
            if (!form) return;
            
            const dateInput = form.querySelector('input[name="departureDate"]');
            const participantsInput = form.querySelector('input[name="participants"]');
            const priceDisplay = form.querySelector('.package-price-display');
            
            if (packageValue && packages[packageValue]) {
                const packageInfo = packages[packageValue];
                
                if (participantsInput) {
                    participantsInput.min = packageInfo.minParticipants;
                    participantsInput.max = packageInfo.maxParticipants;
                    
                    if (parseInt(participantsInput.value) < packageInfo.minParticipants) {
                        participantsInput.value = packageInfo.minParticipants;
                    } else if (parseInt(participantsInput.value) > packageInfo.maxParticipants) {
                        participantsInput.value = packageInfo.maxParticipants;
                    }
                }
                
                if (dateInput) {
                    updateAvailableDates(dateInput, packageInfo.availableDates);
                }
                
                if (priceDisplay) {
                    updatePriceDisplay(priceDisplay, packageInfo.price, participantsInput ? parseInt(participantsInput.value) : 1);
                }
            }
        });
    });
    
    participantsInputs.forEach(input => {
        input.addEventListener('change', function() {
            const form = this.closest('form');
            
            if (!form) return;
            
            const packageSelect = form.querySelector('select[name="package"]');
            const priceDisplay = form.querySelector('.package-price-display');
            
            if (packageSelect && packageSelect.value && packages[packageSelect.value] && priceDisplay) {
                const packageInfo = packages[packageSelect.value];
                updatePriceDisplay(priceDisplay, packageInfo.price, parseInt(this.value));
            }
        });
    });
    
    function updateAvailableDates(dateInput, availableDates) {
        const currentMonth = new Date().getMonth();
        const currentYear = new Date().getFullYear();
        
        const availableDatesArray = [];
        
        for (let month = currentMonth; month < currentMonth + 6; month++) {
            const actualMonth = month % 12;
            const year = currentYear + Math.floor(month / 12);
            const daysInMonth = new Date(year, actualMonth + 1, 0).getDate();
            
            availableDates.forEach(day => {
                if (day <= daysInMonth) {
                    const dateObj = new Date(year, actualMonth, day);
                    
                    if (dateObj > new Date()) {
                        availableDatesArray.push(dateObj.toISOString().split('T')[0]);
                    }
                }
            });
        }
        
        const datePicker = flatpickr(dateInput, {
            enable: availableDatesArray,
            dateFormat: "Y-m-d",
            minDate: "today",
            disableMobile: "true"
        });
    }
    
    function updatePriceDisplay(priceDisplay, basePrice, participants) {
        const totalPrice = basePrice * participants;
        
        let discount = 0;
        
        if (participants >= 6) {
            discount = 0.15; // 15% untuk 6+ peserta
        } else if (participants >= 4) {
            discount = 0.1; // 10% untuk 4-5 peserta
        }
        
        const discountedPrice = totalPrice * (1 - discount);
        
        priceDisplay.innerHTML = `
            <div class="price-calculation">
                <div class="price-item">
                    <span>Harga Dasar</span>
                    <span>${formatCurrency(basePrice)} × ${participants} orang</span>
                </div>
                <div class="price-item">
                    <span>Total</span>
                    <span>${formatCurrency(totalPrice)}</span>
                </div>
                ${discount > 0 ? `
                <div class="price-item discount">
                    <span>Diskon Grup ${discount * 100}%</span>
                    <span>- ${formatCurrency(totalPrice * discount)}</span>
                </div>
                ` : ''}
                <div class="price-item total">
                    <span>Total Pembayaran</span>
                    <span>${formatCurrency(discountedPrice)}</span>
                </div>
            </div>
        `;
    }
    
    function formatCurrency(amount) {
        return new Intl.NumberFormat('id-ID', { style: 'currency', currency: 'IDR' }).format(amount);
    }
}

function initDestinationFilter() {
    const destinationFilterButton = document.createElement('button');
    destinationFilterButton.className = 'destination-filter-button';
    destinationFilterButton.innerHTML = '<i class="fas fa-sliders-h"></i> Filter';
    
    const destinationSection = document.getElementById('destinasi');
    
    if (!destinationSection) return;
    
    const sectionTitle = destinationSection.querySelector('.section-title');
    
    if (sectionTitle) {
        sectionTitle.appendChild(destinationFilterButton);
    }
    
    const filterModal = document.createElement('div');
    filterModal.className = 'modal';
    filterModal.id = 'modal-destination-filter';
    
    filterModal.innerHTML = `
        <div class="modal-content">
            <span class="close-modal" onclick="closeModal('modal-destination-filter')">&times;</span>
            <div class="modal-header">
                <h2>Filter Destinasi</h2>
            </div>
            <div class="modal-body">
                <form class="filter-form">
                    <div class="filter-section">
                        <h3>Jenis Destinasi</h3>
                        <div class="checkbox-group">
                            <input type="checkbox" id="filter-gunung" name="type" value="gunung" checked>
                            <label for="filter-gunung">Gunung</label>
                            
                            <input type="checkbox" id="filter-laut" name="type" value="laut" checked>
                            <label for="filter-laut">Laut</label>
                            
                            <input type="checkbox" id="filter-hutan" name="type" value="hutan" checked>
                            <label for="filter-hutan">Hutan</label>
                            
                            <input type="checkbox" id="filter-danau" name="type" value="danau" checked>
                            <label for="filter-danau">Danau</label>
                        </div>
                    </div>
                    
                    <div class="filter-section">
                        <h3>Lokasi</h3>
                        <div class="checkbox-group">
                            <input type="checkbox" id="filter-jawa" name="location" value="jawa" checked>
                            <label for="filter-jawa">Jawa</label>
                            
                            <input type="checkbox" id="filter-sumatera" name="location" value="sumatera" checked>
                            <label for="filter-sumatera">Sumatera</label>
                            
                            <input type="checkbox" id="filter-kalimantan" name="location" value="kalimantan" checked>
                            <label for="filter-kalimantan">Kalimantan</label>
                            
                            <input type="checkbox" id="filter-sulawesi" name="location" value="sulawesi" checked>
                            <label for="filter-sulawesi">Sulawesi</label>
                            
                            <input type="checkbox" id="filter-bali-nt" name="location" value="bali-nt" checked>
                            <label for="filter-bali-nt">Bali & Nusa Tenggara</label>
                            
                            <input type="checkbox" id="filter-maluku-papua" name="location" value="maluku-papua" checked>
                            <label for="filter-maluku-papua">Maluku & Papua</label>
                        </div>
                    </div>
                    
                    <div class="filter-section">
                        <h3>Tingkat Kesulitan</h3>
                        <div class="range-filter">
                            <input type="range" id="filter-difficulty" name="difficulty" min="1" max="5" value="5">
                            <div class="range-labels">
                                <span>Mudah</span>
                                <span>Sulit</span>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button class="btn-reset-filter">Reset Filter</button>
                <button class="btn-apply-filter">Terapkan Filter</button>
            </div>
        </div>
    `;
    


document.addEventListener('DOMContentLoaded', function() {
    // Initialize range slider with visuals
    const difficultySlider = document.getElementById('filter-difficulty');
    if (difficultySlider) {
        // Create range dots
        const rangeFilter = difficultySlider.closest('.range-filter');
        const rangeDots = document.createElement('div');
        rangeDots.className = 'range-dots';
        
        for (let i = 0; i < 5; i++) {
            const dot = document.createElement('div');
            dot.className = 'range-dot';
            if (i < difficultySlider.value) {
                dot.classList.add('active');
            }
            rangeDots.appendChild(dot);
        }
        
        // Create difficulty text display
        const difficultyText = document.createElement('div');
        difficultyText.className = 'difficulty-text';
        updateDifficultyText(difficultySlider.value, difficultyText);
        
        // Insert elements after range slider
        difficultySlider.insertAdjacentElement('afterend', rangeDots);
        rangeDots.insertAdjacentElement('afterend', difficultyText);
        
        // Update on change
        difficultySlider.addEventListener('input', function() {
            // Update dots
            const dots = rangeDots.querySelectorAll('.range-dot');
            dots.forEach((dot, index) => {
                if (index < this.value) {
                    dot.classList.add('active');
                } else {
                    dot.classList.remove('active');
                }
            });
            
            // Update text
            updateDifficultyText(this.value, difficultyText);
            
            // Update visual fill using CSS variable
            this.style.setProperty('--range-value', this.value);
        });
    }
    
    function updateDifficultyText(value, element) {
        const difficultyLabels = [
            "Sangat Mudah - Cocok untuk pemula",
            "Mudah - Pengalaman dasar diperlukan",
            "Sedang - Diperlukan kebugaran yang cukup",
            "Sulit - Untuk pendaki berpengalaman",
            "Sangat Sulit - Memerlukan persiapan khusus"
        ];
        element.textContent = difficultyLabels[value - 1];
    }
});

    
    document.body.appendChild(filterModal);
    
    destinationFilterButton.addEventListener('click', function() {
        openModal('modal-destination-filter');
    });
    
    const applyFilterButton = filterModal.querySelector('.btn-apply-filter');
    const resetFilterButton = filterModal.querySelector('.btn-reset-filter');
    
    if (applyFilterButton && resetFilterButton) {
        applyFilterButton.addEventListener('click', function() {
            applyDestinationFilter();
            closeModal('modal-destination-filter');
        });
        
        resetFilterButton.addEventListener('click', function() {
            resetDestinationFilter();
        });
    }
    
    function applyDestinationFilter() {
        const typeCheckboxes = document.querySelectorAll('input[name="type"]:checked');
        const locationCheckboxes = document.querySelectorAll('input[name="location"]:checked');
        const difficultyValue = document.getElementById('filter-difficulty').value;
        
        const selectedTypes = Array.from(typeCheckboxes).map(cb => cb.value);
        const selectedLocations = Array.from(locationCheckboxes).map(cb => cb.value);
        
        const destinationCards = document.querySelectorAll('.destination-card');
        
        destinationCards.forEach(card => {
            const cardType = getDestinationType(card);
            const cardLocation = getDestinationLocation(card);
            const cardDifficulty = getDestinationDifficulty(card);
            
            const matchesType = selectedTypes.includes(cardType);
            const matchesLocation = selectedLocations.includes(cardLocation);
            const matchesDifficulty = cardDifficulty <= difficultyValue;
            
            if (matchesType && matchesLocation && matchesDifficulty) {
                card.style.display = 'block';
            } else {
                card.style.display = 'none';
            }
        });
    }
    
    function resetDestinationFilter() {
        const typeCheckboxes = document.querySelectorAll('input[name="type"]');
        const locationCheckboxes = document.querySelectorAll('input[name="location"]');
        const difficultyRange = document.getElementById('filter-difficulty');
        
        typeCheckboxes.forEach(cb => {
            cb.checked = true;
        });
        
        locationCheckboxes.forEach(cb => {
            cb.checked = true;
        });
        
        if (difficultyRange) {
            difficultyRange.value = 5;
        }
        
        const destinationCards = document.querySelectorAll('.destination-card');
        
        destinationCards.forEach(card => {
            card.style.display = 'block';
        });
    }
    
    function getDestinationType(card) {
        const title = card.querySelector('h3').textContent.toLowerCase();
        
        if (title.includes('gunung')) return 'gunung';
        if (title.includes('laut') || title.includes('pantai') || title.includes('pulau')) return 'laut';
        if (title.includes('hutan') || title.includes('taman nasional')) return 'hutan';
        if (title.includes('danau')) return 'danau';
        
        return 'lain';
    }
    
    function getDestinationLocation(card) {
        const location = card.querySelector('.destination-meta span').textContent.toLowerCase();
        
        if (location.includes('jawa')) return 'jawa';
        if (location.includes('sumatera')) return 'sumatera';
        if (location.includes('kalimantan')) return 'kalimantan';
        if (location.includes('sulawesi')) return 'sulawesi';
        if (location.includes('bali') || location.includes('lombok') || location.includes('ntt') || location.includes('ntb')) return 'bali-nt';
        if (location.includes('maluku') || location.includes('papua')) return 'maluku-papua';
        
        return 'lain';
    }
    function getDestinationDifficulty(card) {
        const title = card.querySelector('h3').textContent.toLowerCase();
        
        // Assign difficulty levels based on the destination name
        if (title.includes('rinjani') || title.includes('semeru') || title.includes('kerinci')) {
            return 5; // Most difficult
        } else if (title.includes('bromo') || title.includes('gede') || title.includes('lawu')) {
            return 3; // Medium difficulty
        } else if (title.includes('danau') || title.includes('pantai') || title.includes('taman')) {
            return 1; // Easy
        }
        
        // Default medium difficulty
        return 3;
    }
}

function initPackageComparison() {
    const packageCards = document.querySelectorAll('.package-card');
    
    if (packageCards.length < 2) return;
    
    packageCards.forEach(card => {
        const compareCheckbox = document.createElement('div');
        compareCheckbox.className = 'compare-checkbox';
        compareCheckbox.innerHTML = `
            <input type="checkbox" id="compare-${card.dataset.id || Math.random().toString(36).substr(2, 9)}" class="compare-toggle">
            <label>Bandingkan</label>
        `;
        
        card.appendChild(compareCheckbox);
    });
    
    const compareButton = document.createElement('button');
    compareButton.className = 'compare-packages-button';
    compareButton.innerHTML = 'Bandingkan Paket <span>0</span>';
    compareButton.style.display = 'none';
    
    const packagesSection = document.getElementById('paket');
    
    if (packagesSection) {
        packagesSection.appendChild(compareButton);
    }
    
    const compareToggles = document.querySelectorAll('.compare-toggle');
    
    compareToggles.forEach(toggle => {
        toggle.addEventListener('change', function() {
            updateCompareButton();
        });
    });
    
    compareButton.addEventListener('click', function() {
        openComparisonModal();
    });
    
    function updateCompareButton() {
        const selectedCount = document.querySelectorAll('.compare-toggle:checked').length;
        
        if (selectedCount >= 2) {
            compareButton.style.display = 'block';
            compareButton.querySelector('span').textContent = selectedCount;
        } else {
            compareButton.style.display = 'none';
        }
    }
    
    function openComparisonModal() {
        const selectedPackages = [];
        
        document.querySelectorAll('.compare-toggle:checked').forEach(toggle => {
            const card = toggle.closest('.package-card');
            
            if (card) {
                selectedPackages.push({
                    title: card.querySelector('h3').textContent,
                    image: card.querySelector('img').src,
                    price: card.querySelector('.price').textContent,
                    duration: card.querySelector('.duration').textContent,
                    features: Array.from(card.querySelectorAll('.package-features li')).map(li => li.textContent.trim())
                });
            }
        });
        
        if (selectedPackages.length < 2) return;
        
        const comparisonModal = document.createElement('div');
        comparisonModal.className = 'modal';
        comparisonModal.id = 'modal-package-comparison';
        
        let comparisonContent = `
            <div class="modal-content wide-modal">
                <span class="close-modal" onclick="closeModal('modal-package-comparison')">&times;</span>
                <div class="modal-header">
                    <h2>Perbandingan Paket</h2>
                </div>
                <div class="modal-body">
                    <div class="comparison-table-container">
                        <table class="comparison-table">
                            <thead>
                                <tr>
                                    <th>Fitur</th>
        `;
        
        selectedPackages.forEach(pack => {
            comparisonContent += `<th>${pack.title}</th>`;
        });
        
        comparisonContent += `
                                </tr>
                            </thead>
                            <tbody>
                                <tr class="comparison-images">
                                    <td>Gambar</td>
        `;
        
        selectedPackages.forEach(pack => {
            comparisonContent += `<td><img src="${pack.image}" alt="${pack.title}"></td>`;
        });
        
        comparisonContent += `
                                </tr>
                                <tr>
                                    <td>Harga</td>
        `;
        
        selectedPackages.forEach(pack => {
            comparisonContent += `<td>${pack.price}</td>`;
        });
        
        comparisonContent += `
                                </tr>
                                <tr>
                                    <td>Durasi</td>
        `;
        
        selectedPackages.forEach(pack => {
            comparisonContent += `<td>${pack.duration}</td>`;
        });
        
        // Get all unique features
        const allFeatures = new Set();
        
        selectedPackages.forEach(pack => {
            pack.features.forEach(feature => {
                allFeatures.add(feature);
            });
        });
        
        // Add feature rows
        allFeatures.forEach(feature => {
            comparisonContent += `
                <tr>
                    <td>${feature}</td>
            `;
            
            selectedPackages.forEach(pack => {
                if (pack.features.includes(feature)) {
                    comparisonContent += `<td><i class="fas fa-check text-success"></i></td>`;
                } else {
                    comparisonContent += `<td><i class="fas fa-times text-danger"></i></td>`;
                }
            });
            
            comparisonContent += `</tr>`;
        });
        
        comparisonContent += `
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn-clear-selection" onclick="clearComparisonSelection()">Hapus Pilihan</button>
                </div>
            </div>
        `;
        
        comparisonModal.innerHTML = comparisonContent;
        
        if (document.getElementById('modal-package-comparison')) {
            document.getElementById('modal-package-comparison').remove();
        }
        
        document.body.appendChild(comparisonModal);
        
        openModal('modal-package-comparison');
    }
    
    window.clearComparisonSelection = function() {
        document.querySelectorAll('.compare-toggle:checked').forEach(toggle => {
            toggle.checked = false;
        });
        
        updateCompareButton();
        closeModal('modal-package-comparison');
    };
}

function initFavorites() {
    const favoriteButtons = [];
    
    // Add favorite buttons to destination cards
    document.querySelectorAll('.destination-card').forEach(card => {
        const favoriteButton = document.createElement('button');
        favoriteButton.className = 'favorite-button';
        favoriteButton.innerHTML = '<i class="far fa-heart"></i>';
        
        favoriteButton.addEventListener('click', function(e) {
            e.stopPropagation();
            
            this.classList.toggle('active');
            
            if (this.classList.contains('active')) {
                this.innerHTML = '<i class="fas fa-heart"></i>';
                addToFavorites('destination', card);
            } else {
                this.innerHTML = '<i class="far fa-heart"></i>';
                removeFromFavorites('destination', card);
            }
        });
        
        card.appendChild(favoriteButton);
        favoriteButtons.push(favoriteButton);
    });
    
    // Add favorite buttons to package cards
    document.querySelectorAll('.package-card').forEach(card => {
        const favoriteButton = document.createElement('button');
        favoriteButton.className = 'favorite-button';
        favoriteButton.innerHTML = '<i class="far fa-heart"></i>';
        
        favoriteButton.addEventListener('click', function(e) {
            e.stopPropagation();
            
            this.classList.toggle('active');
            
            if (this.classList.contains('active')) {
                this.innerHTML = '<i class="fas fa-heart"></i>';
                addToFavorites('package', card);
            } else {
                this.innerHTML = '<i class="far fa-heart"></i>';
                removeFromFavorites('package', card);
            }
        });
        
        card.appendChild(favoriteButton);
        favoriteButtons.push(favoriteButton);
    });
    
    // Create favorites icon in header
    const favoritesIcon = document.createElement('div');
    favoritesIcon.className = 'favorites-icon';
    favoritesIcon.innerHTML = '<i class="far fa-heart"></i><span class="count">0</span>';
    
    const header = document.querySelector('header .header-container');
    
    if (header) {
        header.appendChild(favoritesIcon);
    }
    
    favoritesIcon.addEventListener('click', function() {
        openFavoritesModal();
    });
    
    // Load favorites from localStorage
    loadFavorites();
    
    function addToFavorites(type, card) {
        let favorites = JSON.parse(localStorage.getItem('campidn_favorites')) || { destinations: [], packages: [] };
        
        const title = card.querySelector('h3').textContent;
        const image = card.querySelector('img').src;
        
        if (type === 'destination') {
            const location = card.querySelector('.destination-meta span').textContent;
            
            favorites.destinations.push({
                title,
                image,
                location,
                timestamp: Date.now()
            });
        } else if (type === 'package') {
            const price = card.querySelector('.price').textContent;
            const duration = card.querySelector('.duration').textContent;
            
            favorites.packages.push({
                title,
                image,
                price,
                duration,
                timestamp: Date.now()
            });
        }
        
        localStorage.setItem('campidn_favorites', JSON.stringify(favorites));
        updateFavoritesCount();
        
        showNotification('Ditambahkan ke Favorit', `${title} telah ditambahkan ke daftar favorit Anda.`, 'success');
    }
    
    function removeFromFavorites(type, card) {
        let favorites = JSON.parse(localStorage.getItem('campidn_favorites')) || { destinations: [], packages: [] };
        
        const title = card.querySelector('h3').textContent;
        
        if (type === 'destination') {
            favorites.destinations = favorites.destinations.filter(item => item.title !== title);
        } else if (type === 'package') {
            favorites.packages = favorites.packages.filter(item => item.title !== title);
        }
        
        localStorage.setItem('campidn_favorites', JSON.stringify(favorites));
        updateFavoritesCount();
    }
    
    function loadFavorites() {
        const favorites = JSON.parse(localStorage.getItem('campidn_favorites')) || { destinations: [], packages: [] };
        
        // Mark active favorites
        document.querySelectorAll('.destination-card').forEach(card => {
            const title = card.querySelector('h3').textContent;
            const isFavorite = favorites.destinations.some(dest => dest.title === title);
            
            if (isFavorite) {
                const favoriteButton = card.querySelector('.favorite-button');
                if (favoriteButton) {
                    favoriteButton.classList.add('active');
                    favoriteButton.innerHTML = '<i class="fas fa-heart"></i>';
                }
            }
        });
        
        document.querySelectorAll('.package-card').forEach(card => {
            const title = card.querySelector('h3').textContent;
            const isFavorite = favorites.packages.some(pack => pack.title === title);
            
            if (isFavorite) {
                const favoriteButton = card.querySelector('.favorite-button');
                if (favoriteButton) {
                    favoriteButton.classList.add('active');
                    favoriteButton.innerHTML = '<i class="fas fa-heart"></i>';
                }
            }
        });
        
        updateFavoritesCount();
    }
    
    function updateFavoritesCount() {
        const favorites = JSON.parse(localStorage.getItem('campidn_favorites')) || { destinations: [], packages: [] };
        const totalCount = favorites.destinations.length + favorites.packages.length;
        
        const countElement = document.querySelector('.favorites-icon .count');
        
        if (countElement) {
            countElement.textContent = totalCount;
            
            if (totalCount > 0) {
                countElement.style.display = 'block';
            } else {
                countElement.style.display = 'none';
            }
        }
    }
    
    function openFavoritesModal() {
        const favorites = JSON.parse(localStorage.getItem('campidn_favorites')) || { destinations: [], packages: [] };
        
        const favoritesModal = document.createElement('div');
        favoritesModal.className = 'modal';
        favoritesModal.id = 'modal-favorites';
        
        let modalContent = `
            <div class="modal-content">
                <span class="close-modal" onclick="closeModal('modal-favorites')">&times;</span>
                <div class="modal-header">
                    <h2>Favorit Saya</h2>
                </div>
                <div class="modal-body">
        `;
        
        if (favorites.destinations.length === 0 && favorites.packages.length === 0) {
            modalContent += `
                <div class="no-favorites">
                    <i class="far fa-heart"></i>
                    <h3>Belum ada favorit</h3>
                    <p>Tandai destinasi atau paket favorit Anda dengan menekan ikon hati</p>
                </div>
            `;
        } else {
            if (favorites.destinations.length > 0) {
                modalContent += `
                    <div class="favorites-section">
                        <h3>Destinasi Favorit</h3>
                        <div class="favorites-grid">
                `;
                
                favorites.destinations.forEach(dest => {
                    modalContent += `
                        <div class="favorite-item">
                            <img src="${dest.image}" alt="${dest.title}">
                            <div class="favorite-info">
                                <h4>${dest.title}</h4>
                                <p>${dest.location}</p>
                                <button class="btn-remove-favorite" onclick="removeFromFavoritesModal('destination', '${dest.title}')">Hapus</button>
                            </div>
                        </div>
                    `;
                });
                
                modalContent += `
                        </div>
                    </div>
                `;
            }
            
            if (favorites.packages.length > 0) {
                modalContent += `
                    <div class="favorites-section">
                        <h3>Paket Favorit</h3>
                        <div class="favorites-grid">
                `;
                
                favorites.packages.forEach(pack => {
                    modalContent += `
                        <div class="favorite-item">
                            <img src="${pack.image}" alt="${pack.title}">
                            <div class="favorite-info">
                                <h4>${pack.title}</h4>
                                <p>${pack.price} ${pack.duration}</p>
                                <button class="btn-remove-favorite" onclick="removeFromFavoritesModal('package', '${pack.title}')">Hapus</button>
                            </div>
                        </div>
                    `;
                });
                
                modalContent += `
                        </div>
                    </div>
                `;
            }
        }
        
        modalContent += `
                </div>
            </div>
        `;
        
        favoritesModal.innerHTML = modalContent;
        
        if (document.getElementById('modal-favorites')) {
            document.getElementById('modal-favorites').remove();
        }
        
        document.body.appendChild(favoritesModal);
        
        openModal('modal-favorites');
    }
    
    window.removeFromFavoritesModal = function(type, title) {
        let favorites = JSON.parse(localStorage.getItem('campidn_favorites')) || { destinations: [], packages: [] };
        
        if (type === 'destination') {
            favorites.destinations = favorites.destinations.filter(item => item.title !== title);
            
            document.querySelectorAll('.destination-card').forEach(card => {
                const cardTitle = card.querySelector('h3').textContent;
                
                if (cardTitle === title) {
                    const favoriteButton = card.querySelector('.favorite-button');
                    if (favoriteButton) {
                        favoriteButton.classList.remove('active');
                        favoriteButton.innerHTML = '<i class="far fa-heart"></i>';
                    }
                }
            });
        } else if (type === 'package') {
            favorites.packages = favorites.packages.filter(item => item.title !== title);
            
            document.querySelectorAll('.package-card').forEach(card => {
                const cardTitle = card.querySelector('h3').textContent;
                
                if (cardTitle === title) {
                    const favoriteButton = card.querySelector('.favorite-button');
                    if (favoriteButton) {
                        favoriteButton.classList.remove('active');
                        favoriteButton.innerHTML = '<i class="far fa-heart"></i>';
                    }
                }
            });
        }
        
        localStorage.setItem('campidn_favorites', JSON.stringify(favorites));
        updateFavoritesCount();
        
        // Remove item from modal
        const favoriteItem = document.querySelector(`.favorite-item:has(h4:contains('${title}'))`);
        
        if (favoriteItem) {
            favoriteItem.classList.add('removing');
            
            setTimeout(() => {
                favoriteItem.remove();
                
                // Check if section is empty
                const section = document.querySelector(`.favorites-section:has(.favorites-grid:empty)`);
                
                if (section) {
                    section.remove();
                }
                
                // Check if all favorites are gone
                if (favorites.destinations.length === 0 && favorites.packages.length === 0) {
                    const modalBody = document.querySelector('#modal-favorites .modal-body');
                    
                    if (modalBody) {
                        modalBody.innerHTML = `
                            <div class="no-favorites">
                                <i class="far fa-heart"></i>
                                <h3>Belum ada favorit</h3>
                                <p>Tandai destinasi atau paket favorit Anda dengan menekan ikon hati</p>
                            </div>
                        `;
                    }
                }
            }, 300);
        }
    };
}

function initShareButtons() {
    // Add share buttons to cards
    document.querySelectorAll('.destination-card, .package-card, .blog-post').forEach(item => {
        const shareButton = document.createElement('button');
        shareButton.className = 'share-button';
        shareButton.innerHTML = '<i class="fas fa-share-alt"></i>';
        
        shareButton.addEventListener('click', function(e) {
            e.stopPropagation();
            
            const title = item.querySelector('h3').textContent;
            const description = item.querySelector('p').textContent;
            
            openShareModal(title, description);
        });
        
        item.appendChild(shareButton);
    });
    
    function openShareModal(title, description) {
        const shareModal = document.createElement('div');
        shareModal.className = 'modal';
        shareModal.id = 'modal-share';
        
        shareModal.innerHTML = `
            <div class="modal-content">
                <span class="close-modal" onclick="closeModal('modal-share')">&times;</span>
                <div class="modal-header">
                    <h2>Bagikan</h2>
                </div>
                <div class="modal-body">
                    <h3>${title}</h3>
                    <p>${description.substring(0, 100)}...</p>
                    
                    <div class="share-options">
                        <button class="share-option facebook">
                            <i class="fab fa-facebook-f"></i>
                            <span>Facebook</span>
                        </button>
                        <button class="share-option twitter">
                            <i class="fab fa-twitter"></i>
                            <span>Twitter</span>
                        </button>
                        <button class="share-option whatsapp">
                            <i class="fab fa-whatsapp"></i>
                            <span>WhatsApp</span>
                        </button>
                        <button class="share-option telegram">
                            <i class="fab fa-telegram-plane"></i>
                            <span>Telegram</span>
                        </button>
                        <button class="share-option email">
                            <i class="fas fa-envelope"></i>
                            <span>Email</span>
                        </button>
                    </div>
                    
                    <div class="copy-link">
                        <input type="text" value="https://www.campidn.com/destinasi/${encodeURIComponent(title.toLowerCase().replace(/\s+/g, '-'))}" readonly>
                        <button class="btn-copy">Salin</button>
                    </div>
                </div>
            </div>
        `;
        
        if (document.getElementById('modal-share')) {
            document.getElementById('modal-share').remove();
        }
        
        document.body.appendChild(shareModal);
        
        const copyButton = shareModal.querySelector('.btn-copy');
        const linkInput = shareModal.querySelector('.copy-link input');
        
        if (copyButton && linkInput) {
            copyButton.addEventListener('click', function() {
                linkInput.select();
                document.execCommand('copy');
                
                this.textContent = 'Tersalin';
                
                setTimeout(() => {
                    this.textContent = 'Salin';
                }, 2000);
            });
        }
        
        const shareOptions = shareModal.querySelectorAll('.share-option');
        
        shareOptions.forEach(option => {
            option.addEventListener('click', function() {
                const url = `https://www.campidn.com/destinasi/${encodeURIComponent(title.toLowerCase().replace(/\s+/g, '-'))}`;
                
                if (this.classList.contains('facebook')) {
                    window.open(`https://www.facebook.com/sharer/sharer.php?u=${url}`, '_blank');
                } else if (this.classList.contains('twitter')) {
                    window.open(`https://twitter.com/intent/tweet?text=${encodeURIComponent(title)}&url=${url}`, '_blank');
                } else if (this.classList.contains('whatsapp')) {
                    window.open(`https://wa.me/?text=${encodeURIComponent(title + ' ' + url)}`, '_blank');
                } else if (this.classList.contains('telegram')) {
                    window.open(`https://t.me/share/url?url=${url}&text=${encodeURIComponent(title)}`, '_blank');
                } else if (this.classList.contains('email')) {
                    window.location.href = `mailto:?subject=${encodeURIComponent(title)}&body=${encodeURIComponent(description + '\n\n' + url)}`;
                }
                
                closeModal('modal-share');
            });
        });
        
        openModal('modal-share');
    }
}

function initRatingSystem() {
    const testimonialItems = document.querySelectorAll('.testimonial-item');
    
    testimonialItems.forEach(item => {
        const ratingContainer = item.querySelector('.rating');
        
        if (ratingContainer) {
            const rating = countStars(ratingContainer);
            
            // Update with interactive stars
            ratingContainer.innerHTML = '';
            
            for (let i = 1; i <= 5; i++) {
                const star = document.createElement('i');
                
                if (i <= rating) {
                    star.className = 'fas fa-star';
                } else if (i - 0.5 <= rating) {
                    star.className = 'fas fa-star-half-alt';
                } else {
                    star.className = 'far fa-star';
                }
                
                ratingContainer.appendChild(star);
            }
        }
    });
    
    // Add rating to review form
    const reviewForm = document.getElementById('reviewForm');
    
    if (reviewForm) {
        const ratingFieldset = document.createElement('fieldset');
        ratingFieldset.className = 'rating-fieldset';
        ratingFieldset.innerHTML = `
            <legend>Beri Rating</legend>
            <div class="star-rating">
                <input type="radio" id="star5" name="rating" value="5"><label for="star5"></label>
                <input type="radio" id="star4" name="rating" value="4"><label for="star4"></label>
                <input type="radio" id="star3" name="rating" value="3"><label for="star3"></label>
                <input type="radio" id="star2" name="rating" value="2"><label for="star2"></label>
                <input type="radio" id="star1" name="rating" value="1"><label for="star1"></label>
            </div>
        `;
        
        const firstFormGroup = reviewForm.querySelector('.form-group');
        
        if (firstFormGroup) {
            reviewForm.insertBefore(ratingFieldset, firstFormGroup);
        }
    }
    
    function countStars(container) {
        const fullStars = container.querySelectorAll('.fa-star').length;
        const halfStars = container.querySelectorAll('.fa-star-half').length;
        
        return fullStars + (halfStars * 0.5);
    }
}

function initAccordions() {
    const accordionContainers = document.querySelectorAll('.accordion');
    
    accordionContainers.forEach(container => {
        const accordionItems = container.querySelectorAll('.accordion-item');
        
        accordionItems.forEach(item => {
            const header = item.querySelector('.accordion-header');
            const content = item.querySelector('.accordion-content');
            
            if (header && content) {
                header.addEventListener('click', function() {
                    const isActive = item.classList.contains('active');
                    
                    // Close all other items
                    if (container.dataset.behavior === 'single') {
                        accordionItems.forEach(otherItem => {
                            if (otherItem !== item) {
                                otherItem.classList.remove('active');
                                const otherContent = otherItem.querySelector('.accordion-content');
                                
                                if (otherContent) {
                                    otherContent.style.maxHeight = null;
                                }
                            }
                        });
                    }
                    
                    // Toggle current item
                    item.classList.toggle('active');
                    
                    if (item.classList.contains('active')) {
                        content.style.maxHeight = content.scrollHeight + 'px';
                    } else {
                        content.style.maxHeight = null;
                    }
                });
                
                // Open first item if specified
                if (container.dataset.openFirst === 'true' && item === accordionItems[0]) {
                    item.classList.add('active');
                    content.style.maxHeight = content.scrollHeight + 'px';
                }
            }
        });
    });
    
    // Create accordions for itinerary
    const itineraries = document.querySelectorAll('.itinerary');
    
    itineraries.forEach(itinerary => {
        if (!itinerary.classList.contains('accordion')) {
            itinerary.classList.add('accordion');
            itinerary.dataset.behavior = 'multi';
            
            const days = itinerary.querySelectorAll('.itinerary-day');
            
            days.forEach(day => {
                day.classList.add('accordion-item');
                
                const header = day.querySelector('.day-header');
                const content = day.querySelector('.day-content');
                
                if (header && content) {
                    header.classList.add('accordion-header');
                    content.classList.add('accordion-content');
                    
                    // Add toggle icon
                    const toggleIcon = document.createElement('span');
                    toggleIcon.className = 'accordion-toggle';
                    toggleIcon.innerHTML = '<i class="fas fa-chevron-down"></i>';
                    
                    header.appendChild(toggleIcon);
                }
            });
        }
    });
    
    // Initialize newly created accordions
    initAccordions();
}

function initCounters() {
    const counterElements = document.querySelectorAll('.counter');
    
    if (!counterElements.length) return;
    
    const counterObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const counter = entry.target;
                const target = parseInt(counter.dataset.target);
                const duration = parseInt(counter.dataset.duration) || 2000;
                const increment = target / (duration / 16);
                
                let current = 0;
                
                const updateCounter = () => {
                    current += increment;
                    
                    if (current < target) {
                        counter.textContent = Math.ceil(current);
                        requestAnimationFrame(updateCounter);
                    } else {
                        counter.textContent = target;
                    }
                };
                
                updateCounter();
                counterObserver.unobserve(counter);
            }
        });
    }, { threshold: 0.5 });
    
    counterElements.forEach(counter => {
        counterObserver.observe(counter);
    });
}

function initValidation() {
    const forms = document.querySelectorAll('form');
    
    forms.forEach(form => {
        const requiredFields = form.querySelectorAll('[required]');
        
        requiredFields.forEach(field => {
            // Add asterisk to labels
            const fieldId = field.id;
            const label = form.querySelector(`label[for="${fieldId}"]`);
            
            if (label && !label.innerHTML.includes('*')) {
                label.innerHTML += ' <span class="required">*</span>';
            }
            
            // Add validation styles and events
            field.addEventListener('blur', function() {
                validateField(this);
            });
            
            field.addEventListener('input', function() {
                if (this.classList.contains('invalid')) {
                    validateField(this);
                }
            });
        });
    });
    
    function validateField(field) {
        if (!field.value.trim()) {
            field.classList.add('invalid');
            
            let errorMsg = field.parentElement.querySelector('.error-message');
            
            if (!errorMsg) {
                errorMsg = document.createElement('span');
                errorMsg.className = 'error-message';
                field.parentElement.appendChild(errorMsg);
            }
            
// Melanjutkan dari potongan terakhir di script.js
errorMsg.textContent = 'Bidang ini wajib diisi';
} else {
    field.classList.remove('invalid');
    
    const errorMsg = field.parentElement.querySelector('.error-message');
    if (errorMsg) {
        errorMsg.remove();
    }
    
    // Validasi tambahan berdasarkan tipe input
    if (field.type === 'email' && !validateEmail(field.value)) {
        field.classList.add('invalid');
        
        let errorMsg = field.parentElement.querySelector('.error-message');
        
        if (!errorMsg) {
            errorMsg = document.createElement('span');
            errorMsg.className = 'error-message';
            field.parentElement.appendChild(errorMsg);
        }
        
        errorMsg.textContent = 'Format email tidak valid';
    } else if (field.type === 'tel' && !validatePhone(field.value)) {
        field.classList.add('invalid');
        
        let errorMsg = field.parentElement.querySelector('.error-message');
        
        if (!errorMsg) {
            errorMsg = document.createElement('span');
            errorMsg.className = 'error-message';
            field.parentElement.appendChild(errorMsg);
        }
        
        errorMsg.textContent = 'Format nomor telepon tidak valid';
    }
}
}

function validateEmail(email) {
const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
return regex.test(email);
}

function validatePhone(phone) {
const regex = /^[+]?[\d\s-]{8,15}$/;
return regex.test(phone);
}
}

function initLanguageSelector() {
const languages = [
{ code: 'id', name: 'Indonesia', flag: '🇮🇩' },
{ code: 'en', name: 'English', flag: '🇬🇧' },
{ code: 'zh', name: '中文', flag: '🇨🇳' },
{ code: 'ja', name: '日本語', flag: '🇯🇵' },
{ code: 'ko', name: '한국어', flag: '🇰🇷' }
];

const languageSelector = document.createElement('div');
languageSelector.className = 'language-selector';

const currentLanguage = document.createElement('div');
currentLanguage.className = 'current-language';
currentLanguage.innerHTML = `${languages[0].flag} ${languages[0].name} <i class="fas fa-chevron-down"></i>`;

const languageDropdown = document.createElement('div');
languageDropdown.className = 'language-dropdown';

languages.forEach(lang => {
const langOption = document.createElement('div');
langOption.className = 'language-option';
langOption.dataset.lang = lang.code;
langOption.innerHTML = `${lang.flag} ${lang.name}`;

langOption.addEventListener('click', function() {
    changeLanguage(lang.code);
    currentLanguage.innerHTML = `${lang.flag} ${lang.name} <i class="fas fa-chevron-down"></i>`;
    languageDropdown.classList.remove('show');
});

languageDropdown.appendChild(langOption);
});

languageSelector.appendChild(currentLanguage);
languageSelector.appendChild(languageDropdown);

const header = document.querySelector('header .header-container');

if (header) {
header.appendChild(languageSelector);
}

currentLanguage.addEventListener('click', function(e) {
e.stopPropagation();
languageDropdown.classList.toggle('show');
});

document.addEventListener('click', function(e) {
if (!languageSelector.contains(e.target)) {
    languageDropdown.classList.remove('show');
}
});

function changeLanguage(langCode) {
// Implementasi pergantian bahasa
console.log(`Changed language to: ${langCode}`);

// Demo saja: menampilkan notifikasi
showNotification('Bahasa Diubah', `Bahasa situs telah diubah ke ${languages.find(l => l.code === langCode).name}`, 'info');

// Dalam implementasi nyata, mungkin perlu reload halaman atau mengubah teks dengan AJAX
}
}

function initCurrencyConverter() {
const currencies = [
{ code: 'IDR', symbol: 'Rp', name: 'Indonesian Rupiah' },
{ code: 'USD', symbol: '$', name: 'US Dollar' },
{ code: 'EUR', symbol: '€', name: 'Euro' },
{ code: 'SGD', symbol: 'S$', name: 'Singapore Dollar' },
{ code: 'AUD', symbol: 'A$', name: 'Australian Dollar' }
];

const exchangeRates = {
IDR: 1,
USD: 15500,
EUR: 18000,
SGD: 11500,
AUD: 10000
};

const currencySelector = document.createElement('div');
currencySelector.className = 'currency-selector';

const currentCurrency = document.createElement('div');
currentCurrency.className = 'current-currency';
currentCurrency.innerHTML = `${currencies[0].symbol} ${currencies[0].code} <i class="fas fa-chevron-down"></i>`;

const currencyDropdown = document.createElement('div');
currencyDropdown.className = 'currency-dropdown';

currencies.forEach(currency => {
const currencyOption = document.createElement('div');
currencyOption.className = 'currency-option';
currencyOption.dataset.currency = currency.code;
currencyOption.innerHTML = `${currency.symbol} ${currency.code} - ${currency.name}`;

currencyOption.addEventListener('click', function() {
    changeCurrency(currency.code);
    currentCurrency.innerHTML = `${currency.symbol} ${currency.code} <i class="fas fa-chevron-down"></i>`;
    currencyDropdown.classList.remove('show');
});

currencyDropdown.appendChild(currencyOption);
});

currencySelector.appendChild(currentCurrency);
currencySelector.appendChild(currencyDropdown);

const header = document.querySelector('header .header-container');

if (header) {
header.appendChild(currencySelector);
}

currentCurrency.addEventListener('click', function(e) {
e.stopPropagation();
currencyDropdown.classList.toggle('show');
});

document.addEventListener('click', function(e) {
if (!currencySelector.contains(e.target)) {
    currencyDropdown.classList.remove('show');
}
});

function changeCurrency(currencyCode) {
const currentCurrency = localStorage.getItem('campidn_currency') || 'IDR';

if (currencyCode === currentCurrency) return;

const conversionRate = exchangeRates[currentCurrency] / exchangeRates[currencyCode];

// Update semua harga
const priceElements = document.querySelectorAll('.price');

priceElements.forEach(element => {
    const originalPrice = element.getAttribute('data-original-price');
    
    if (!originalPrice) {
        // Menyimpan harga asli (dalam IDR) untuk konversi berikutnya
        const priceText = element.textContent.trim();
        const price = parseFloat(priceText.replace(/[^\d]/g, ''));
        
        if (!isNaN(price)) {
            element.setAttribute('data-original-price', price);
            
            const convertedPrice = Math.round(price / exchangeRates[currencyCode]);
            const formattedPrice = formatCurrency(convertedPrice, currencyCode);
            
            element.textContent = formattedPrice;
        }
    } else {
        const originalPriceValue = parseFloat(originalPrice);
        const convertedPrice = Math.round(originalPriceValue / exchangeRates[currencyCode]);
        const formattedPrice = formatCurrency(convertedPrice, currencyCode);
        
        element.textContent = formattedPrice;
    }
});

localStorage.setItem('campidn_currency', currencyCode);

showNotification('Mata Uang Diubah', `Mata uang telah diubah ke ${currencies.find(c => c.code === currencyCode).name}`, 'info');
}

function formatCurrency(amount, currencyCode) {
const currency = currencies.find(c => c.code === currencyCode);

if (currencyCode === 'IDR') {
    return `${currency.symbol} ${amount.toLocaleString('id-ID')}`;
} else {
    return `${currency.symbol} ${amount.toLocaleString('en-US')}`;
}
}

// Load currency preference from localStorage
const savedCurrency = localStorage.getItem('campidn_currency');

if (savedCurrency && savedCurrency !== 'IDR') {
const currency = currencies.find(c => c.code === savedCurrency);

if (currency) {
    currentCurrency.innerHTML = `${currency.symbol} ${currency.code} <i class="fas fa-chevron-down"></i>`;
    changeCurrency(savedCurrency);
}
}
}

function initWeatherWidget() {
const weatherData = {
'rinjani': { temp: 18, condition: 'partly-cloudy', humidity: 74, wind: 12 },
'bromo': { temp: 16, condition: 'cloudy', humidity: 80, wind: 15 },
'rajaampat': { temp: 29, condition: 'sunny', humidity: 65, wind: 8 },
'toba': { temp: 24, condition: 'rainy', humidity: 85, wind: 10 },
'komodo': { temp: 31, condition: 'sunny', humidity: 60, wind: 14 },
'wakatobi': { temp: 28, condition: 'partly-cloudy', humidity: 70, wind: 9 }
};

const destinationDetails = document.querySelectorAll('.modal-description');

destinationDetails.forEach(detail => {
const modal = detail.closest('.modal');

if (!modal) return;

const modalId = modal.id;
const destinationName = modalId.replace('modal-', '');

if (weatherData[destinationName]) {
    const weather = weatherData[destinationName];
    
    const weatherWidget = document.createElement('div');
    weatherWidget.className = 'weather-widget';
    
    const conditionIcon = getWeatherIcon(weather.condition);
    
    weatherWidget.innerHTML = `
        <h3>Cuaca Terkini</h3>
        <div class="weather-content">
            <div class="weather-main">
                <div class="weather-icon">${conditionIcon}</div>
                <div class="weather-temp">${weather.temp}°C</div>
            </div>
            <div class="weather-details">
                <div class="weather-detail">
                    <i class="fas fa-tint"></i>
                    <span>${weather.humidity}%</span>
                </div>
                <div class="weather-detail">
                    <i class="fas fa-wind"></i>
                    <span>${weather.wind} km/h</span>
                </div>
            </div>
        </div>
        <div class="weather-update">Diperbarui: ${new Date().toLocaleTimeString('id-ID', { hour: '2-digit', minute: '2-digit' })}</div>
    `;
    
    const locationSection = detail.querySelector('h3:contains("Jalur Pendakian"), h3:contains("Destinasi Utama")');
    
    if (locationSection) {
        locationSection.parentNode.insertBefore(weatherWidget, locationSection);
    } else {
        detail.appendChild(weatherWidget);
    }
}
});

function getWeatherIcon(condition) {
switch (condition) {
    case 'sunny':
        return '<i class="fas fa-sun"></i>';
    case 'partly-cloudy':
        return '<i class="fas fa-cloud-sun"></i>';
    case 'cloudy':
        return '<i class="fas fa-cloud"></i>';
    case 'rainy':
        return '<i class="fas fa-cloud-rain"></i>';
    case 'stormy':
        return '<i class="fas fa-bolt"></i>';
    default:
        return '<i class="fas fa-cloud-sun"></i>';
}
}
}

function initCountdownTimers() {
const promoEndDate = new Date();
promoEndDate.setDate(promoEndDate.getDate() + 7); // Promo berakhir dalam 7 hari

const notificationContent = document.querySelector('.notification-content');

if (notificationContent) {
const countdownTimer = document.createElement('div');
countdownTimer.className = 'countdown-timer';

notificationContent.appendChild(countdownTimer);

updateCountdown();

setInterval(updateCountdown, 1000);

function updateCountdown() {
    const now = new Date();
    const diff = promoEndDate - now;
    
    if (diff <= 0) {
        countdownTimer.innerHTML = 'Promo telah berakhir!';
        return;
    }
    
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
    const seconds = Math.floor((diff % (1000 * 60)) / 1000);
    
    countdownTimer.innerHTML = `
        <div class="countdown-item">
            <span class="countdown-value">${days}</span>
            <span class="countdown-label">Hari</span>
        </div>
        <div class="countdown-item">
            <span class="countdown-value">${hours}</span>
            <span class="countdown-label">Jam</span>
        </div>
        <div class="countdown-item">
            <span class="countdown-value">${minutes}</span>
            <span class="countdown-label">Menit</span>
        </div>
        <div class="countdown-item">
            <span class="countdown-value">${seconds}</span>
            <span class="countdown-label">Detik</span>
        </div>
    `;
}
}

// Countdown untuk booking deadlines
const bookingDeadlines = document.querySelectorAll('.booking-deadline');

bookingDeadlines.forEach(deadline => {
const dateStr = deadline.getAttribute('data-deadline');

if (!dateStr) return;

const deadlineDate = new Date(dateStr);

updateBookingDeadline(deadline, deadlineDate);

setInterval(() => {
    updateBookingDeadline(deadline, deadlineDate);
}, 60000); // Update setiap menit
});

function updateBookingDeadline(element, deadlineDate) {
const now = new Date();
const diff = deadlineDate - now;

if (diff <= 0) {
    element.innerHTML = '<span class="expired">Pendaftaran Ditutup</span>';
    return;
}

const days = Math.floor(diff / (1000 * 60 * 60 * 24));
const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));

if (days > 0) {
    element.innerHTML = `<span class="deadline-time">Sisa ${days} hari ${hours} jam</span>`;
} else {
    const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
    element.innerHTML = `<span class="deadline-time urgent">Sisa ${hours} jam ${minutes} menit</span>`;
}
}
}

function initTips() {
const tips = [
{
    title: 'Tips Mendaki Gunung',
    content: 'Selalu bawa perlengkapan medis dasar dan informasikan rencana perjalanan Anda kepada teman atau keluarga.',
    icon: 'fas fa-mountain'
},
{
    title: 'Tips Snorkeling',
    content: 'Gunakan tabir surya yang aman untuk terumbu karang dan jangan menyentuh kehidupan laut.',
    icon: 'fas fa-water'
},
{
    title: 'Tips Berkemah',
    content: 'Bawa kantong sampah ekstra dan selalu tinggalkan area perkemahan lebih bersih dari sebelumnya.',
    icon: 'fas fa-campground'
},
{
    title: 'Tips Fotografi',
    content: 'Golden hour (1 jam setelah matahari terbit atau sebelum terbenam) adalah waktu terbaik untuk foto pemandangan.',
    icon: 'fas fa-camera'
},
{
    title: 'Tips Hemat',
    content: 'Pesan paket perjalanan jauh-jauh hari dan pertimbangkan untuk pergi di low season untuk harga terbaik.',
    icon: 'fas fa-coins'
}
];

// Pilih tip secara acak
const randomTip = tips[Math.floor(Math.random() * tips.length)];

// Buat tip card
const tipCard = document.createElement('div');
tipCard.className = 'tip-card';

tipCard.innerHTML = `
<div class="tip-icon">
    <i class="${randomTip.icon}"></i>
</div>
<div class="tip-content">
    <h4>${randomTip.title}</h4>
    <p>${randomTip.content}</p>
</div>
<button class="close-tip"><i class="fas fa-times"></i></button>
`;

// Tambahkan tip card ke halaman
document.body.appendChild(tipCard);

// Tampilkan tip card setelah 30 detik
setTimeout(() => {
tipCard.classList.add('show');
}, 30000);

// Tutup tip card saat tombol close diklik
const closeButton = tipCard.querySelector('.close-tip');

if (closeButton) {
closeButton.addEventListener('click', function() {
    tipCard.classList.remove('show');
    
    setTimeout(() => {
        tipCard.remove();
    }, 300);
});
}

// Tips dalam bentuk tooltip
const tooltipElements = document.querySelectorAll('[data-tooltip]');

tooltipElements.forEach(element => {
element.addEventListener('mouseenter', function() {
    const tooltipText = this.getAttribute('data-tooltip');
    
    const tooltip = document.createElement('div');
    tooltip.className = 'tooltip';
    tooltip.textContent = tooltipText;
    
    document.body.appendChild(tooltip);
    
    const rect = this.getBoundingClientRect();
    const tooltipHeight = tooltip.offsetHeight;
    
    tooltip.style.left = rect.left + (rect.width / 2) - (tooltip.offsetWidth / 2) + 'px';
    tooltip.style.top = rect.top - tooltipHeight - 10 + window.scrollY + 'px';
    
    tooltip.classList.add('show');
    
    this.addEventListener('mouseleave', function tooltipRemove() {
        tooltip.classList.remove('show');
        
        setTimeout(() => {
            tooltip.remove();
        }, 300);
        
        this.removeEventListener('mouseleave', tooltipRemove);
    });
});
});
}

function initPrint() {
// Tambahkan tombol print di halaman detail
const detailPages = document.querySelectorAll('.modal-content');

detailPages.forEach(page => {
const header = page.querySelector('.modal-header');

if (!header) return;

const printButton = document.createElement('button');
printButton.className = 'print-button';
printButton.innerHTML = '<i class="fas fa-print"></i> Cetak';

printButton.addEventListener('click', function() {
    printContent(page);
});

header.appendChild(printButton);
});

function printContent(content) {
const printWindow = window.open('', '_blank');

if (!printWindow) {
    alert('Mohon izinkan popup untuk mencetak halaman');
    return;
}

const title = content.querySelector('h2').textContent;
const contentClone = content.cloneNode(true);

// Hapus elemen yang tidak perlu dicetak
const removeElements = contentClone.querySelectorAll('.close-modal, .modal-footer, .print-button, .share-button, .favorite-button');

removeElements.forEach(element => {
    element.remove();
});

const printStyles = `
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            padding: 20px;
        }
        
        h2 {
            color: #0c5460;
            border-bottom: 2px solid #0c5460;
            padding-bottom: 10px;
            margin-bottom: 20px;
        }
        
        h3 {
            color: #0c5460;
            margin-top: 20px;
            margin-bottom: 10px;
        }
        
        img {
            max-width: 100%;
            height: auto;
        }
        
        .modal-content, .modal-header, .modal-body {
            width: 100%;
            max-width: 100%;
            margin: 0;
            padding: 0;
            background: none;
            box-shadow: none;
        }
        
        .print-header, .print-footer {
            text-align: center;
            margin: 20px 0;
        }
        
        .print-logo {
            font-weight: bold;
            font-size: 24px;
            color: #0c5460;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        
        th {
            background-color: #f2f2f2;
        }
        
        .print-date {
            text-align: right;
            font-size: 12px;
            color: #666;
            margin-top: 30px;
        }
        
        @media print {
            body {
                padding: 0;
            }
        }
    </style>
`;

const now = new Date();
const dateFormatted = now.toLocaleDateString('id-ID', { day: '2-digit', month: 'long', year: 'numeric' });

const printHeader = `
    <div class="print-header">
        <div class="print-logo">Camp IDN</div>
        <div>Jl. Petualang No. 123, Jakarta Selatan, Indonesia</div>
        <div>info@campidn.com | +62 812 3456 7890</div>
    </div>
`;

const printFooter = `
    <div class="print-footer">
        <div class="print-date">Dicetak pada: ${dateFormatted}</div>
        <div>© ${new Date().getFullYear()} Camp IDN. All Rights Reserved.</div>
    </div>
`;

const printHTML = `
    <!DOCTYPE html>
    <html>
    <head>
        <title>${title} - Camp IDN</title>
        ${printStyles}
    </head>
    <body>
        ${printHeader}
        <h2>${title}</h2>
        ${contentClone.querySelector('.modal-body').innerHTML}
        ${printFooter}
        <script>
            window.onload = function() {
                setTimeout(function() {
                    window.print();
                    window.onfocus = function() {
                        window.close();
                    };
                }, 500);
            };
        </script>
    </body>
    </html>
`;

printWindow.document.open();
printWindow.document.write(printHTML);
printWindow.document.close();
}
}

function initThemeSwitcher() {
const themes = [
{ name: 'default', icon: 'fas fa-sun', label: 'Tema Cerah' },
{ name: 'dark', icon: 'fas fa-moon', label: 'Tema Gelap' },
{ name: 'nature', icon: 'fas fa-leaf', label: 'Tema Alam' },
{ name: 'ocean', icon: 'fas fa-water', label: 'Tema Laut' }
];

const themeSwitcher = document.createElement('div');
themeSwitcher.className = 'theme-switcher';

const currentTheme = document.createElement('button');
currentTheme.className = 'current-theme';
currentTheme.innerHTML = `<i class="${themes[0].icon}"></i>`;
currentTheme.setAttribute('data-tooltip', themes[0].label);

const themeOptions = document.createElement('div');
themeOptions.className = 'theme-options';

themes.forEach(theme => {
const themeOption = document.createElement('button');
themeOption.className = 'theme-option';
themeOption.setAttribute('data-theme', theme.name);
themeOption.innerHTML = `<i class="${theme.icon}"></i>`;
themeOption.setAttribute('data-tooltip', theme.label);

themeOption.addEventListener('click', function() {
    changeTheme(theme.name);
    currentTheme.innerHTML = `<i class="${theme.icon}"></i>`;
    currentTheme.setAttribute('data-tooltip', theme.label);
    themeOptions.classList.remove('show');
});

themeOptions.appendChild(themeOption);
});

themeSwitcher.appendChild(currentTheme);
themeSwitcher.appendChild(themeOptions);

const header = document.querySelector('header .header-container');

if (header) {
header.appendChild(themeSwitcher);
}

currentTheme.addEventListener('click', function() {
themeOptions.classList.toggle('show');
});

document.addEventListener('click', function(e) {
if (!themeSwitcher.contains(e.target)) {
    themeOptions.classList.remove('show');
}
});

function changeTheme(themeName) {
document.body.className = '';
document.body.classList.add(`theme-${themeName}`);
localStorage.setItem('campidn_theme', themeName);

showNotification('Tema Diubah', `Tema situs telah diubah ke ${themes.find(t => t.name === themeName).label}`, 'info');
}

// Load theme preference from localStorage
const savedTheme = localStorage.getItem('campidn_theme');

if (savedTheme) {
const theme = themes.find(t => t.name === savedTheme);

if (theme) {
    document.body.classList.add(`theme-${savedTheme}`);
    currentTheme.innerHTML = `<i class="${theme.icon}"></i>`;
    currentTheme.setAttribute('data-tooltip', theme.label);
}
}

// Dark mode detection
const prefersDarkMode = window.matchMedia('(prefers-color-scheme: dark)').matches;

if (prefersDarkMode && !savedTheme) {
const darkTheme = themes.find(t => t.name === 'dark');

if (darkTheme) {
    document.body.classList.add('theme-dark');
    currentTheme.innerHTML = `<i class="${darkTheme.icon}"></i>`;
    currentTheme.setAttribute('data-tooltip', darkTheme.label);
}
}
}

function initAccessibility() {
const accessibilityButton = document.createElement('button');
accessibilityButton.className = 'accessibility-button';
accessibilityButton.innerHTML = '<i class="fas fa-universal-access"></i>';
accessibilityButton.setAttribute('data-tooltip', 'Opsi Aksesibilitas');

document.body.appendChild(accessibilityButton);

const accessibilityMenu = document.createElement('div');
accessibilityMenu.className = 'accessibility-menu';

accessibilityMenu.innerHTML = `
<h3>Opsi Aksesibilitas</h3>
<div class="accessibility-options">
    <div class="accessibility-option">
        <label for="font-size">Ukuran Teks</label>
        <div class="option-buttons">
            <button data-action="decrease-font"><i class="fas fa-minus"></i></button>
            <button data-action="reset-font"><i class="fas fa-undo"></i></button>
            <button data-action="increase-font"><i class="fas fa-plus"></i></button>
        </div>
    </div>
    <div class="accessibility-option">
        <label for="contrast">Kontras Tinggi</label>
                <div class="option-buttons">
                    <button data-action="toggle-contrast" class="toggle-button">
                        <span class="toggle-indicator"></span>
                    </button>
                </div>
            </div>
            <div class="accessibility-option">
                <label for="animations">Nonaktifkan Animasi</label>
                <div class="option-buttons">
                    <button data-action="toggle-animations" class="toggle-button">
                        <span class="toggle-indicator"></span>
                    </button>
                </div>
            </div>
            <div class="accessibility-option">
                <label for="dyslexic">Font Ramah Disleksia</label>
                <div class="option-buttons">
                    <button data-action="toggle-dyslexic" class="toggle-button">
                        <span class="toggle-indicator"></span>
                    </button>
                </div>
            </div>
            <div class="accessibility-option">
                <label for="focus">Indikator Fokus</label>
                <div class="option-buttons">
                    <button data-action="toggle-focus" class="toggle-button">
                        <span class="toggle-indicator"></span>
                    </button>
                </div>
            </div>
        </div>
        <button class="close-accessibility">Tutup</button>
    `;
    
    document.body.appendChild(accessibilityMenu);
    
    accessibilityButton.addEventListener('click', function() {
        accessibilityMenu.classList.toggle('show');
    });
    
    const closeButton = accessibilityMenu.querySelector('.close-accessibility');
    
    if (closeButton) {
        closeButton.addEventListener('click', function() {
            accessibilityMenu.classList.remove('show');
        });
    }
    
    // Ukuran Font
    const decreaseFontButton = accessibilityMenu.querySelector('[data-action="decrease-font"]');
    const resetFontButton = accessibilityMenu.querySelector('[data-action="reset-font"]');
    const increaseFontButton = accessibilityMenu.querySelector('[data-action="increase-font"]');
    
    if (decreaseFontButton && resetFontButton && increaseFontButton) {
        decreaseFontButton.addEventListener('click', function() {
            changeFontSize(-1);
        });
        
        resetFontButton.addEventListener('click', function() {
            resetFontSize();
        });
        
        increaseFontButton.addEventListener('click', function() {
            changeFontSize(1);
        });
    }
    
    // Toggle Buttons
    const toggleButtons = accessibilityMenu.querySelectorAll('.toggle-button');
    
    toggleButtons.forEach(button => {
        button.addEventListener('click', function() {
            this.classList.toggle('active');
            
            const action = this.getAttribute('data-action');
            
            switch (action) {
                case 'toggle-contrast':
                    toggleHighContrast();
                    break;
                case 'toggle-animations':
                    toggleAnimations();
                    break;
                case 'toggle-dyslexic':
                    toggleDyslexicFont();
                    break;
                case 'toggle-focus':
                    toggleFocusIndicator();
                    break;
            }
        });
    });
    
    // Load saved preferences
    loadAccessibilityPreferences();
    
    function changeFontSize(step) {
        const html = document.documentElement;
        let currentSize = parseFloat(getComputedStyle(html).fontSize);
        
        if (isNaN(currentSize)) {
            currentSize = 16; // Default font size
        }
        
        const newSize = currentSize + step;
        
        if (newSize >= 12 && newSize <= 24) {
            html.style.fontSize = newSize + 'px';
            localStorage.setItem('campidn_fontSize', newSize);
        }
    }
    
    function resetFontSize() {
        document.documentElement.style.fontSize = '';
        localStorage.removeItem('campidn_fontSize');
    }
    
    function toggleHighContrast() {
        document.body.classList.toggle('high-contrast');
        localStorage.setItem('campidn_highContrast', document.body.classList.contains('high-contrast'));
    }
    
    function toggleAnimations() {
        document.body.classList.toggle('no-animations');
        localStorage.setItem('campidn_noAnimations', document.body.classList.contains('no-animations'));
    }
    
    function toggleDyslexicFont() {
        document.body.classList.toggle('dyslexic-font');
        localStorage.setItem('campidn_dyslexicFont', document.body.classList.contains('dyslexic-font'));
    }
    
    function toggleFocusIndicator() {
        document.body.classList.toggle('focus-indicator');
        localStorage.setItem('campidn_focusIndicator', document.body.classList.contains('focus-indicator'));
    }
    
    function loadAccessibilityPreferences() {
        // Font Size
        const fontSize = localStorage.getItem('campidn_fontSize');
        
        if (fontSize) {
            document.documentElement.style.fontSize = fontSize + 'px';
        }
        
        // High Contrast
        if (localStorage.getItem('campidn_highContrast') === 'true') {
            document.body.classList.add('high-contrast');
            const contrastButton = accessibilityMenu.querySelector('[data-action="toggle-contrast"]');
            
            if (contrastButton) {
                contrastButton.classList.add('active');
            }
        }
        
        // No Animations
        if (localStorage.getItem('campidn_noAnimations') === 'true') {
            document.body.classList.add('no-animations');
            const animationsButton = accessibilityMenu.querySelector('[data-action="toggle-animations"]');
            
            if (animationsButton) {
                animationsButton.classList.add('active');
            }
        }
        
        // Dyslexic Font
        if (localStorage.getItem('campidn_dyslexicFont') === 'true') {
            document.body.classList.add('dyslexic-font');
            const dyslexicButton = accessibilityMenu.querySelector('[data-action="toggle-dyslexic"]');
            
            if (dyslexicButton) {
                dyslexicButton.classList.add('active');
            }
        }
        
        // Focus Indicator
        if (localStorage.getItem('campidn_focusIndicator') === 'true') {
            document.body.classList.add('focus-indicator');
            const focusButton = accessibilityMenu.querySelector('[data-action="toggle-focus"]');
            
            if (focusButton) {
                focusButton.classList.add('active');
            }
        }
    }
    
    // Keyboard Navigation
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Tab') {
            document.body.classList.add('keyboard-navigation');
        }
    });
    
    document.addEventListener('mousedown', function() {
        document.body.classList.remove('keyboard-navigation');
    });
}

// Main gallery initialization
document.addEventListener('DOMContentLoaded', function() {
    // Initialize gallery features
    initGalleryFilter();
    initImageGallery();
    initLoadMoreButton();
});

// Filter gallery items by category
function initGalleryFilter() {
    const filterButtons = document.querySelectorAll('.filter-btn');
    const galleryItems = document.querySelectorAll('.gallery-item');
    
    // Initial state - show only first 8 items
    updateVisibleItems(8);
    
    // Add click event to filter buttons
    filterButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all buttons
            filterButtons.forEach(btn => btn.classList.remove('active'));
            
            // Add active class to clicked button
            this.classList.add('active');
            
            // Reset visible items count when filter changes
            resetVisibleItems();
        });
    });
}

// Image gallery with lightbox
function initImageGallery() {
    // Get all gallery images
    const galleryImages = document.querySelectorAll('.gallery-item img');
    
    // Return if no images found
    if (!galleryImages.length) return;
    
    // Get lightbox container
    const lightbox = document.getElementById('lightbox-gallery');
    
    // Initialize lightbox elements
    const lightboxImg = lightbox.querySelector('.lightbox-content img');
    const lightboxTitle = lightbox.querySelector('.lightbox-details h3');
    const lightboxDescription = lightbox.querySelector('.lightbox-details p');
    const lightboxThumbnails = lightbox.querySelector('.lightbox-thumbnails');
    const prevButton = lightbox.querySelector('.lightbox-prev');
    const nextButton = lightbox.querySelector('.lightbox-next');
    const closeButton = lightbox.querySelector('.lightbox-close');
    
    // Track current image index
    let currentIndex = 0;
    
    // Add click handlers to gallery images to open lightbox
    galleryImages.forEach((img) => {
        img.addEventListener('click', function() {
            // Find the containing gallery item
            const clickedItem = this.closest('.gallery-item');
            
            // Get all visible gallery items (this accounts for filtering)
            const visibleItems = Array.from(document.querySelectorAll('.gallery-item.visible'));
            
            // Find the index of the clicked item within visible items
            const visibleIndex = visibleItems.findIndex(item => item === clickedItem);
            
            // Only proceed if the item is found among visible items
            if (visibleIndex === -1) return;
            
            // Open lightbox
            lightbox.classList.add('active');
            document.body.classList.add('lightbox-open');
            
            // Empty thumbnails container before adding new thumbnails
            lightboxThumbnails.innerHTML = '';
            
            // Create thumbnails only for visible items
            visibleItems.forEach((item, idx) => {
                const itemImg = item.querySelector('img');
                const thumbnail = document.createElement('div');
                thumbnail.className = 'lightbox-thumbnail';
                thumbnail.innerHTML = `<img src="${itemImg.src}" alt="${itemImg.alt}">`;
                
                // Add click handler for thumbnails
                thumbnail.addEventListener('click', function() {
                    currentIndex = idx;
                    updateLightbox(visibleItems);
                });
                
                lightboxThumbnails.appendChild(thumbnail);
            });
            
            // Set current index to the clicked item
            currentIndex = visibleIndex;
            updateLightbox(visibleItems);
            
            // Next image handler 
            nextButton.onclick = function() {
                currentIndex = (currentIndex + 1) % visibleItems.length;
                updateLightbox(visibleItems);
            };
            
            // Previous image handler
            prevButton.onclick = function() {
                currentIndex = (currentIndex - 1 + visibleItems.length) % visibleItems.length;
                updateLightbox(visibleItems);
            };
            
            // Close lightbox handler
            closeButton.onclick = function() {
                lightbox.classList.remove('active');
                document.body.classList.remove('lightbox-open');
            };
            
            // Keyboard navigation handler
            const keyHandler = function(e) {
                if (!lightbox.classList.contains('active')) return;
                
                if (e.key === 'Escape') {
                    lightbox.classList.remove('active');
                    document.body.classList.remove('lightbox-open');
                    document.removeEventListener('keydown', keyHandler);
                } else if (e.key === 'ArrowRight') {
                    currentIndex = (currentIndex + 1) % visibleItems.length;
                    updateLightbox(visibleItems);
                } else if (e.key === 'ArrowLeft') {
                    currentIndex = (currentIndex - 1 + visibleItems.length) % visibleItems.length;
                    updateLightbox(visibleItems);
                }
            };
            
            // Add keyboard event listener
            document.addEventListener('keydown', keyHandler);
            
            // Touch swipe functionality
            let touchStartX = 0;
            let touchEndX = 0;
            
            // Touch start handler
            const touchStartHandler = function(e) {
                touchStartX = e.changedTouches[0].screenX;
            };
            
            // Touch end handler
            const touchEndHandler = function(e) {
                touchEndX = e.changedTouches[0].screenX;
                
                if (touchEndX < touchStartX - 50) {
                    // Swipe left - next image
                    currentIndex = (currentIndex + 1) % visibleItems.length;
                    updateLightbox(visibleItems);
                } else if (touchEndX > touchStartX + 50) {
                    // Swipe right - previous image
                    currentIndex = (currentIndex - 1 + visibleItems.length) % visibleItems.length;
                    updateLightbox(visibleItems);
                }
            };
            
            // Add touch event listeners
            lightbox.addEventListener('touchstart', touchStartHandler);
            lightbox.addEventListener('touchend', touchEndHandler);
            
            // Remove touch event listeners when lightbox closes
            closeButton.addEventListener('click', function() {
                lightbox.removeEventListener('touchstart', touchStartHandler);
                lightbox.removeEventListener('touchend', touchEndHandler);
            });
        });
    });
    
    // Update lightbox content and UI
    function updateLightbox(visibleItems) {
        const currentItem = visibleItems[currentIndex];
        const img = currentItem.querySelector('img');
        const title = currentItem.querySelector('.gallery-overlay h3');
        const description = currentItem.querySelector('.gallery-overlay p');
        
        // Add fade effect when updating image
        lightboxImg.classList.add('fade');
        
        setTimeout(() => {
            // Update image source and details
            lightboxImg.src = img.src;
            lightboxImg.alt = img.alt;
            
            // Update title if exists
            if (title) {
                lightboxTitle.textContent = title.textContent;
            } else {
                lightboxTitle.textContent = '';
            }
            
            // Update description if exists
            if (description) {
                lightboxDescription.textContent = description.textContent;
            } else {
                lightboxDescription.textContent = '';
            }
            
            // Remove fade effect
            lightboxImg.classList.remove('fade');
        }, 300);
        
        // Update active thumbnail
        const thumbnails = lightboxThumbnails.querySelectorAll('.lightbox-thumbnail');
        
        thumbnails.forEach((thumb, index) => {
            if (index === currentIndex) {
                thumb.classList.add('active');
            } else {
                thumb.classList.remove('active');
            }
        });
    }
}

// Load more functionality
function initLoadMoreButton() {
    const loadMoreBtn = document.getElementById('load-more');
    
    if (!loadMoreBtn) return; // Exit if button not found
    
    // Initial items to show
    let visibleItems = 8;
    const itemIncrement = 8; // Number of items to add when "load more" is clicked
    
    // Hide items beyond initial count
    updateVisibleItems(visibleItems);
    
    // Add click handler to load more button
    loadMoreBtn.addEventListener('click', function(e) {
        e.preventDefault();
        
        // Get current filter
        const activeFilter = document.querySelector('.filter-btn.active').getAttribute('data-filter');
        
        // Get gallery items matching current filter
        const filteredItems = Array.from(document.querySelectorAll('.gallery-item'))
            .filter(item => {
                return activeFilter === 'all' || item.getAttribute('data-category') === activeFilter;
            });
        
        // Calculate new visible items count
        visibleItems += itemIncrement;
        
        // Don't exceed total items count
        if (visibleItems >= filteredItems.length) {
            visibleItems = filteredItems.length;
            loadMoreBtn.style.display = 'none'; // Hide button when all items are shown
        }
        
        // Update visible items
        updateVisibleItems(visibleItems);
    });
}

// Helper function to update visible items
function updateVisibleItems(count) {
    // Get current filter
    const activeFilter = document.querySelector('.filter-btn.active').getAttribute('data-filter');
    
    // Get gallery items
    const galleryItems = document.querySelectorAll('.gallery-item');
    
    let visibleCount = 0;
    
    galleryItems.forEach(item => {
        // Check if item matches current filter
        const matchesFilter = activeFilter === 'all' || item.getAttribute('data-category') === activeFilter;
        
        // Hide all items initially
        item.classList.remove('visible');
        item.style.display = 'none';
        
        // Show items that match filter and are within count limit
        if (matchesFilter) {
            visibleCount++;
            if (visibleCount <= count) {
                item.classList.add('visible');
                item.style.display = 'block';
            }
        }
    });
    
    // Update load more button visibility
    const loadMoreBtn = document.getElementById('load-more');
    
    if (!loadMoreBtn) return;
    
    // Calculate total items matching current filter
    const totalMatchingItems = Array.from(galleryItems).filter(item => {
        return activeFilter === 'all' || item.getAttribute('data-category') === activeFilter;
    }).length;
    
    // Show/hide button based on whether there are more items to show
    if (count >= totalMatchingItems) {
        loadMoreBtn.style.display = 'none';
    } else {
        loadMoreBtn.style.display = 'inline-block';
    }
}

// Reset visible items when filter changes
function resetVisibleItems() {
    // Reset to showing only 8 items
    updateVisibleItems(8);
    
    // Show the load more button (it will be hidden automatically if not needed)
    const loadMoreBtn = document.getElementById('load-more');
    if (loadMoreBtn) {
        loadMoreBtn.style.display = 'inline-block';
    }
}

function initAdvancedFilters() {
    // Membuat filter untuk berbagai bagian situs
    createSortOptions();
    createPriceRangeFilter();
    createAdvancedSearch();
    
    function createSortOptions() {
        // Menambahkan opsi pengurutan untuk paket perjalanan
        const packagesSection = document.querySelector('.packages-section');
        
        if (!packagesSection) return;
        
        const sectionTitle = packagesSection.querySelector('.section-title');
        
        if (!sectionTitle) return;
        
        const sortOptions = document.createElement('div');
        sortOptions.className = 'sort-options';
        
        sortOptions.innerHTML = `
            <label for="sort-packages">Urutkan:</label>
            <select id="sort-packages">
                <option value="default">Urutan Default</option>
                <option value="price-low">Harga: Rendah ke Tinggi</option>
                <option value="price-high">Harga: Tinggi ke Rendah</option>
                <option value="duration-short">Durasi: Terpendek</option>
                <option value="duration-long">Durasi: Terpanjang</option>
                <option value="popular">Popularitas</option>
            </select>
        `;
        
        sectionTitle.appendChild(sortOptions);
        
        const sortSelect = document.getElementById('sort-packages');
        
        if (sortSelect) {
            sortSelect.addEventListener('change', function() {
                sortPackages(this.value);
            });
        }
    }
    
    function createPriceRangeFilter() {
        // Menambahkan filter rentang harga
        const packagesSection = document.querySelector('.packages-section');
        
        if (!packagesSection) return;
        
        const filtersContainer = document.createElement('div');
        filtersContainer.className = 'price-filter-container';
        
       
        
        const packagesContainer = packagesSection.querySelector('.packages-container');
        
        if (packagesContainer) {
            packagesSection.insertBefore(filtersContainer, packagesContainer);
            
            initRangeSliders();
        }
    }
    
    function createAdvancedSearch() {
        // Tambahkan pencarian lanjutan untuk seluruh situs
        const searchIcon = document.querySelector('.search-icon');
        
        if (!searchIcon) return;
        
        const advancedSearchModal = document.createElement('div');
        advancedSearchModal.className = 'modal';
        advancedSearchModal.id = 'modal-advanced-search';
        
        advancedSearchModal.innerHTML = `
            <div class="modal-content">
                <span class="close-modal" onclick="closeModal('modal-advanced-search')">&times;</span>
                <div class="modal-header">
                    <h2>Pencarian Lanjutan</h2>
                </div>
                <div class="modal-body">
                    <form class="advanced-search-form">
                        <div class="form-group">
                            <label for="search-keyword">Kata Kunci</label>
                            <input type="text" id="search-keyword" placeholder="Masukkan kata kunci...">
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="search-category">Kategori</label>
                                <select id="search-category">
                                    <option value="all">Semua</option>
                                    <option value="destination">Destinasi</option>
                                    <option value="package">Paket Perjalanan</option>
                                    <option value="blog">Artikel Blog</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="search-location">Lokasi</label>
                                <select id="search-location">
                                    <option value="all">Semua Lokasi</option>
                                    <option value="jawa">Jawa</option>
                                    <option value="bali-nusa-tenggara">Bali & Nusa Tenggara</option>
                                    <option value="sulawesi">Sulawesi</option>
                                    <option value="kalimantan">Kalimantan</option>
                                    <option value="sumatera">Sumatera</option>
                                    <option value="papua-maluku">Papua & Maluku</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="search-price-min">Harga Minimum</label>
                                <input type="number" id="search-price-min" placeholder="Rp 0" min="0" step="100000">
                            </div>
                            <div class="form-group">
                                <label for="search-price-max">Harga Maksimum</label>
                                <input type="number" id="search-price-max" placeholder="Tidak terbatas" min="0" step="100000">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Jenis Aktivitas</label>
                            <div class="checkbox-group">
                                <label><input type="checkbox" name="activity" value="mountain"> Pendakian Gunung</label>
                                <label><input type="checkbox" name="activity" value="diving"> Diving & Snorkeling</label>
                                <label><input type="checkbox" name="activity" value="rafting"> Rafting & Arung Jeram</label>
                                <label><input type="checkbox" name="activity" value="cultural"> Wisata Budaya</label>
                                <label><input type="checkbox" name="activity" value="camping"> Camping & Glamping</label>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="search-duration">Durasi (Hari)</label>
                            <div class="range-slider">
                                <input type="range" id="search-duration" min="1" max="14" value="14">
                                <div class="range-value">
                                    <span id="duration-value">14</span> hari
                                </div>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn-submit">Cari</button>
                        <button type="reset" class="btn-reset">Reset</button>
                    </form>
                </div>
            </div>
        `;
        
        document.body.appendChild(advancedSearchModal);
        
        // Tambahkan tombol pencarian lanjutan di modal pencarian biasa
        const searchModal = document.getElementById('modal-search');
        
        if (searchModal) {
            const searchContainer = searchModal.querySelector('.search-container');
            
            if (searchContainer) {
                const advancedButton = document.createElement('button');
                advancedButton.className = 'advanced-search-button';
                advancedButton.innerHTML = 'Pencarian Lanjutan <i class="fas fa-sliders-h"></i>';
                
                advancedButton.addEventListener('click', function() {
                    closeModal('modal-search');
                    openModal('modal-advanced-search');
                });
                
                searchContainer.appendChild(advancedButton);
            }
        }
        
        // Inisialisasi durasi range slider
        const durationRange = document.getElementById('search-duration');
        const durationValue = document.getElementById('duration-value');
        
        if (durationRange && durationValue) {
            durationRange.addEventListener('input', function() {
                durationValue.textContent = this.value;
            });
        }
        
        // Handle form submission
        const advancedSearchForm = advancedSearchModal.querySelector('.advanced-search-form');
        
        if (advancedSearchForm) {
            advancedSearchForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                // Collect all form values
                const keyword = document.getElementById('search-keyword').value;
                const category = document.getElementById('search-category').value;
                const location = document.getElementById('search-location').value;
                const priceMin = document.getElementById('search-price-min').value;
                const priceMax = document.getElementById('search-price-max').value;
                const duration = document.getElementById('search-duration').value;
                
                const activities = [];
                document.querySelectorAll('input[name="activity"]:checked').forEach(checkbox => {
                    activities.push(checkbox.value);
                });
                
                // Perform search (mock for demo)
                performAdvancedSearch({
                    keyword,
                    category,
                    location,
                    priceMin,
                    priceMax,
                    activities,
                    duration
                });
            });
        }
    }
    
    function sortPackages(sortType) {
        const packagesContainer = document.querySelector('.packages-container');
        
        if (!packagesContainer) return;
        
        const packages = Array.from(packagesContainer.querySelectorAll('.package-card'));
        
        // Sort packages based on selected option
        packages.sort((a, b) => {
            if (sortType === 'price-low') {
                const priceA = getPackagePrice(a);
                const priceB = getPackagePrice(b);
                return priceA - priceB;
            } else if (sortType === 'price-high') {
                const priceA = getPackagePrice(a);
                const priceB = getPackagePrice(b);
                return priceB - priceA;
            } else if (sortType === 'duration-short') {
                const durationA = getPackageDuration(a);
                const durationB = getPackageDuration(b);
                return durationA - durationB;
            } else if (sortType === 'duration-long') {
                const durationA = getPackageDuration(a);
                const durationB = getPackageDuration(b);
                return durationB - durationA;
            } else if (sortType === 'popular') {
                // Untuk demo, asumsikan paket dengan tag "Populer" lebih populer
                const popularA = a.querySelector('.package-tag')?.textContent === 'Populer' ? 1 : 0;
                const popularB = b.querySelector('.package-tag')?.textContent === 'Populer' ? 1 : 0;
                return popularB - popularA;
            } else {
                // Default sorting - return to original order
                return packages.indexOf(a) - packages.indexOf(b);
            }
        });
        
        // Reappend sorted packages
        packages.forEach(pkg => {
            packagesContainer.appendChild(pkg);
        });
        
        // Add animation effect
        packages.forEach((pkg, index) => {
            setTimeout(() => {
                pkg.classList.add('sorted');
                
                setTimeout(() => {
                    pkg.classList.remove('sorted');
                }, 500);
            }, index * 100);
        });
    }
    
    function filterPackagesByPrice(minPrice, maxPrice) {
        const packagesContainer = document.querySelector('.packages-container');
        
        if (!packagesContainer) return;
        
        const packages = packagesContainer.querySelectorAll('.package-card');
        
        packages.forEach(pkg => {
            const packagePrice = getPackagePrice(pkg);
            
            if (packagePrice >= minPrice && packagePrice <= maxPrice) {
                pkg.style.display = 'block';
            } else {
                pkg.style.display = 'none';
            }
        });
    }
    
    function initRangeSliders() {
        const minSlider = document.getElementById('price-range-min');
        const maxSlider = document.getElementById('price-range-max');
        const minValue = document.getElementById('price-min-value');
        const maxValue = document.getElementById('price-max-value');
        const applyButton = document.getElementById('apply-price-filter');
        
        if (!minSlider || !maxSlider || !minValue || !maxValue || !applyButton) return;
        
        // Update values on slider change
        minSlider.addEventListener('input', function() {
            const min = parseInt(this.value);
            const max = parseInt(maxSlider.value);
            
            if (min > max) {
                this.value = max;
                minValue.textContent = formatCurrency(max);
            } else {
                minValue.textContent = formatCurrency(min);
            }
        });
        
        maxSlider.addEventListener('input', function() {
            const min = parseInt(minSlider.value);
            const max = parseInt(this.value);
            
            if (max < min) {
                this.value = min;
                maxValue.textContent = formatCurrency(min);
            } else {
                maxValue.textContent = formatCurrency(max);
            }
        });
        
        // Apply filter
        applyButton.addEventListener('click', function() {
            const minPrice = parseInt(minSlider.value);
            const maxPrice = parseInt(maxSlider.value);
            
            filterPackagesByPrice(minPrice, maxPrice);
        });
    }
    
    function getPackagePrice(packageElement) {
        const priceText = packageElement.querySelector('.price').textContent.trim();
        // Extract numbers from price text (e.g. "Rp 5.500.000" -> 5500000)
        return parseInt(priceText.replace(/[^\d]/g, ''));
    }
    
    function getPackageDuration(packageElement) {
        const durationText = packageElement.querySelector('.duration').textContent.trim();
        // Extract numbers from duration text (e.g. "/ orang (5 hari)" -> 5)
        const match = durationText.match(/\((\d+)\s*hari\)/);
        return match ? parseInt(match[1]) : 0;
    }
    
    function formatCurrency(amount) {
        return `Rp ${amount.toLocaleString('id-ID')}`;
    }
    
    function performAdvancedSearch(searchParams) {
        // Close modal
        closeModal('modal-advanced-search');
        
        // Show loading indicator
        showSearchResults('loading');
        
        // Simulate server delay
        setTimeout(() => {
            // For demo purposes, we'll randomly decide if we have results
            const hasResults = Math.random() > 0.3;
            
            if (hasResults) {
                showSearchResults('results', searchParams);
            } else {
                showSearchResults('no-results', searchParams);
            }
        }, 1500);
    }
    
    function showSearchResults(state, searchParams = {}) {
        // Create results modal if it doesn't exist
        let resultsModal = document.getElementById('modal-search-results');
        
        if (!resultsModal) {
            resultsModal = document.createElement('div');
            resultsModal.className = 'modal';
            resultsModal.id = 'modal-search-results';
            document.body.appendChild(resultsModal);
        }
        
        let modalContent = `
            <div class="modal-content wide-modal">
                <span class="close-modal" onclick="closeModal('modal-search-results')">&times;</span>
                <div class="modal-header">
                    <h2>Hasil Pencarian</h2>
                </div>
                <div class="modal-body">
        `;
        
        if (state === 'loading') {
            modalContent += `
                <div class="search-loading">
                    <i class="fas fa-spinner fa-spin"></i>
                    <p>Sedang mencari...</p>
                </div>
            `;
        } else if (state === 'no-results') {
            modalContent += `
                <div class="no-results">
                    <i class="fas fa-search"></i>
                    <h3>Tidak Ditemukan</h3>
                    <p>Maaf, tidak ada hasil untuk pencarian dengan kriteria tersebut. Coba ubah filter pencarian Anda.</p>
                    <button class="btn-primary" onclick="openModal('modal-advanced-search')">Ubah Pencarian</button>
                </div>
            `;
        } else if (state === 'results') {
            // In a real implementation, these would be actual search results from a server
            // For demo, we'll create mock results
            modalContent += `
                <div class="search-summary">
                    <p>Ditemukan <strong>6 hasil</strong> untuk pencarian Anda</p>
                    <div class="search-filters">
                        ${searchParams.keyword ? `<span class="search-filter">Kata Kunci: ${searchParams.keyword} <button class="remove-filter">×</button></span>` : ''}
                        ${searchParams.location !== 'all' ? `<span class="search-filter">Lokasi: ${searchParams.location} <button class="remove-filter">×</button></span>` : ''}
                        ${searchParams.priceMin ? `<span class="search-filter">Harga Min: Rp ${parseInt(searchParams.priceMin).toLocaleString('id-ID')} <button class="remove-filter">×</button></span>` : ''}
                        ${searchParams.priceMax ? `<span class="search-filter">Harga Max: Rp ${parseInt(searchParams.priceMax).toLocaleString('id-ID')} <button class="remove-filter">×</button></span>` : ''}
                        ${searchParams.activities && searchParams.activities.length ? `<span class="search-filter">Aktivitas: ${searchParams.activities.join(', ')} <button class="remove-filter">×</button></span>` : ''}
                        ${searchParams.duration ? `<span class="search-filter">Durasi: ${searchParams.duration} hari <button class="remove-filter">×</button></span>` : ''}
                    </div>
                </div>
                <div class="search-results-grid">
            `;
            
            // Generate mock results
            const resultTypes = ['destination', 'package', 'blog'];
            
            for (let i = 0; i < 6; i++) {
                const resultType = resultTypes[Math.floor(Math.random() * resultTypes.length)];
                const resultTitle = getRandomResultTitle(resultType);
                
                modalContent += `
                    <div class="search-result-item ${resultType}">
                        <img src="/api/placeholder/300/200" alt="${resultTitle}">
                        <div class="result-content">
                            <div class="result-type">${resultType === 'destination' ? 'Destinasi' : resultType === 'package' ? 'Paket' : 'Artikel'}</div>
                            <h3>${resultTitle}</h3>
                            <p>${getRandomResultDescription(resultType)}</p>
                            <div class="result-meta">
                                ${resultType === 'package' ? 
                                    `<span class="result-price">Rp ${(Math.floor(Math.random() * 10) + 1) * 1000000}</span>
                                    <span class="result-duration">${Math.floor(Math.random() * 10) + 1} Hari</span>` : 
                                    resultType === 'destination' ? 
                                    `<span class="result-location"><i class="fas fa-map-marker-alt"></i> ${getRandomLocation()}</span>` :
                                    `<span class="result-date"><i class="far fa-calendar"></i> ${getRandomDate()}</span>`
                                }
                            </div>
                            <a href="#" class="btn-view-details">Lihat Detail</a>
                        </div>
                    </div>
                `;
            }
            
            modalContent += `
                </div>
                <div class="search-pagination">
                    <button class="pagination-prev disabled"><i class="fas fa-chevron-left"></i></button>
                    <div class="pagination-numbers">
                        <span class="active">1</span>
                        <span>2</span>
                        <span>3</span>
                    </div>
                    <button class="pagination-next"><i class="fas fa-chevron-right"></i></button>
                </div>
            `;
        }
        
        modalContent += `
                </div>
            </div>
        `;
        
        resultsModal.innerHTML = modalContent;
        
        // Attach event listeners to filter buttons
        const removeFilterButtons = resultsModal.querySelectorAll('.remove-filter');
        
        removeFilterButtons.forEach(button => {
            button.addEventListener('click', function() {
                const filterItem = this.parentElement;
                filterItem.remove();
                
                // In a real app, this would trigger a new search without the removed filter
                // For demo, we'll just show a notification
                showNotification('Filter Dihapus', 'Hasil pencarian telah diperbarui', 'info');
            });
        });
        
        openModal('modal-search-results');
    }
    
    function getRandomResultTitle(type) {
        const titles = {
            destination: [
                'Gunung Rinjani',
                'Pantai Pink Lombok',
                'Raja Ampat',
                'Taman Nasional Komodo',
                'Danau Toba',
                'Danau Kelimutu',
                'Gunung Bromo',
                'Pulau Derawan',
                'Taman Nasional Ujung Kulon',
                'Kawah Ijen'
            ],
            package: [
                'Ekspedisi Rinjani 3D2N',
                'Diving Raja Ampat 5D4N',
                'Petualangan Komodo 4D3N',
                'Eksotis Derawan 6D5N',
                'Bromo Midnight Tour 2D1N',
                'Pesona Danau Toba 3D2N',
                'Bali Cultural Experience 7D6N',
                'Toraja Heritage Tour 5D4N',
                'Arung Jeram Sungai Ayung 1D',
                'Trekking Merbabu 3D2N'
            ],
            blog: [
                'Tips Mendaki untuk Pemula',
                'Perlengkapan Wajib Diving',
                'Kuliner Khas Raja Ampat',
                'Menjelajah Budaya Toraja',
                'Fotografi Landscape di Bromo',
                'Survival Guide: Hutan Kalimantan',
                'Mengenal Satwa Khas Komodo',
                'Sejarah Danau Toba',
                'Panduan Lengkap Berkemah',
                'Wisata Ramah Lingkungan di Indonesia'
            ]
        };
        
        return titles[type][Math.floor(Math.random() * titles[type].length)];
    }
    
    function getRandomResultDescription(type) {
        const descriptions = {
            destination: [
                'Nikmati keindahan alam yang memukau dengan pemandangan yang spektakuler.',
                'Eksplorasi keanekaragaman hayati yang luar biasa di lokasi ini.',
                'Destinasi populer dengan keindahan alam yang masih terjaga keasliannya.',
                'Surga tersembunyi yang menawarkan pengalaman tak terlupakan.',
                'Objek wisata yang wajib dikunjungi dengan keunikan yang menakjubkan.'
            ],
            package: [
                'Paket petualangan lengkap dengan pemandu berpengalaman dan peralatan berkualitas.',
                'Nikmati pengalaman mendalam dengan akomodasi, transportasi, dan makan yang sudah termasuk.',
                'Paket favorit para traveler dengan nilai terbaik untuk pengalaman yang luar biasa.',
                'Petualangan seru dengan jadwal yang diatur secara profesional untuk kenyamanan Anda.',
                'Paket ekslusif dengan fasilitas premium dan pengalaman yang personal.'
            ],
            blog: [
                'Temukan informasi berharga dan tips berguna dalam artikel ini.',
                'Panduan lengkap dengan informasi terkini untuk perjalanan Anda.',
                'Berbagi pengalaman dan cerita inspiratif dari para petualang.',
                'Tips dan trik dari ahli untuk memaksimalkan pengalaman perjalanan Anda.',
                'Ulasan mendalam yang akan membantu Anda merencanakan petualangan terbaik.'
            ]
        };
        
        return descriptions[type][Math.floor(Math.random() * descriptions[type].length)];
    }
    
    function getRandomLocation() {
        const locations = [
            'Lombok, NTB',
            'Papua Barat',
            'Jawa Timur',
            'Nusa Tenggara Timur',
            'Sumatera Utara',
            'Sulawesi Tenggara',
            'Bali',
            'Maluku',
            'Kalimantan Timur',
            'Yogyakarta'
        ];
        
        return locations[Math.floor(Math.random() * locations.length)];
    }
    
    function getRandomDate() {
        const months = [
            'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
            'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
        ];
        
        const day = Math.floor(Math.random() * 28) + 1;
        const month = months[Math.floor(Math.random() * months.length)];
        const year = 2025;
        
        return `${day} ${month} ${year}`;
    }
}

function initProgressiveLoading() {
    // Implementasi lazy loading konten saat scroll
    // Berguna untuk halaman dengan konten yang banyak
    
    const lazyContainers = document.querySelectorAll('.lazy-load-container');
    
    if (!lazyContainers.length) return;
    
    // Create loading indicator
    const loadingIndicator = document.createElement('div');
    loadingIndicator.className = 'lazy-loading-indicator';
    loadingIndicator.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Memuat lebih banyak...</span>';
    
    let currentPage = 1;
    let isLoading = false;
    
    // Observer untuk mendeteksi saat container visible
    const loadObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting && !isLoading) {
                loadMoreContent(entry.target);
            }
        });
    }, { threshold: 0.5 });
    
    lazyContainers.forEach(container => {
        // Tambahkan loading indicator
        container.appendChild(loadingIndicator.cloneNode(true));
        
        // Observe container
        loadObserver.observe(container);
    });
    
    function loadMoreContent(container) {
        isLoading = true;
        
        // Tampilkan loading indicator
        const indicator = container.querySelector('.lazy-loading-indicator');
        
        if (indicator) {
            indicator.classList.add('active');
        }
        
        // Simulasi request ke server
        setTimeout(() => {
            // Generate konten baru berdasarkan container type
            let newContent = '';
            
            if (container.classList.contains('destinations-container')) {
                newContent = generateDestinationCards(3);
            } else if (container.classList.contains('packages-container')) {
                newContent = generatePackageCards(2);
            } else if (container.classList.contains('blog-container')) {
                newContent = generateBlogPosts(2);
            } else if (container.classList.contains('gallery-container')) {
                newContent = generateGalleryItems(4);
            }
            
            // Insert new content before loading indicator
            if (newContent) {
                indicator.insertAdjacentHTML('beforebegin', newContent);
                
                // Initialize new content elements
                initNewContentElements(container);
                
                currentPage++;
                
                // Remove observer after reaching page limit
                if (currentPage >= 3) {
                    loadObserver.unobserve(container);
                    indicator.remove();
                }
            }
            
            // Hide loading indicator
            if (indicator) {
                indicator.classList.remove('active');
            }
            
            isLoading = false;
        }, 1500);
    }
    
    function generateDestinationCards(count) {
        const destinations = [
            { name: 'Pulau Weh', location: 'Aceh', type: 'laut', feature: 'Diving Spot' },
            { name: 'Bukit Lawang', location: 'Sumatera Utara', type: 'hutan', feature: 'Orangutan' },
            { name: 'Pulau Belitung', location: 'Bangka Belitung', type: 'laut', feature: 'Pantai Batu Granit' },
            { name: 'Ngarai Sianok', location: 'Sumatera Barat', type: 'lembah', feature: '15 km' },
            { name: 'Taman Nasional Bunaken', location: 'Sulawesi Utara', type: 'laut', feature: 'Terumbu Karang' },
            { name: 'Puncak Jaya', location: 'Papua', type: 'gunung', feature: '4.884 mdpl' }
        ];
        
        let html = '';
        
        for (let i = 0; i < count; i++) {
            const index = Math.floor(Math.random() * destinations.length);
            const dest = destinations[index];
            
            html += `
                <div class="destination-card">
                    <img src="/api/placeholder/400/300" alt="${dest.name}">
                    <div class="destination-info">
                        <h3>${dest.name}</h3>
                        <p>Keindahan alam yang menakjubkan dengan pemandangan yang spektakuler.</p>
                        <div class="destination-meta">
                            <span><i class="fas fa-map-marker-alt"></i> ${dest.location}</span>
                            <span><i class="fas fa-${dest.type === 'gunung' ? 'mountain' : dest.type === 'laut' ? 'water' : 'tree'}"></i> ${dest.feature}</span>
                        </div>
                        <button class="btn-view-details" onclick="openModal('modal-detail')">Lihat Detail</button>
                    </div>
                </div>
            `;
        }
        
        return html;
    }
    
    function generatePackageCards(count) {
        const packages = [
            { name: 'Petualangan Derawan', tag: 'Baru', price: '7.500.000', duration: '6 hari' },
            { name: 'Eksplorasi Bunaken', tag: 'Populer', price: '6.800.000', duration: '5 hari' },
            { name: 'Trekking Merbabu', tag: 'Hemat', price: '2.200.000', duration: '3 hari' },
            { name: 'Wisata Budaya Yogyakarta', tag: 'Keluarga', price: '3.500.000', duration: '4 hari' }
        ];
        
        let html = '';
        
        for (let i = 0; i < count; i++) {
            const index = Math.floor(Math.random() * packages.length);
            const pkg = packages[index];
            
            html += `
                <div class="package-card">
                    <div class="package-tag">${pkg.tag}</div>
                    <img src="/api/placeholder/400/250" alt="${pkg.name}">
                    <div class="package-info">
                        <h3>${pkg.name}</h3>
                        <p>Paket petualangan lengkap dengan pengalaman tak terlupakan.</p>
                        <ul class="package-features">
                            <li><i class="fas fa-check"></i> Pemandu berpengalaman</li>
                            <li><i class="fas fa-check"></i> Akomodasi nyaman</li>
                            <li><i class="fas fa-check"></i> Transportasi</li>
                            <li><i class="fas fa-check"></i> Makan 3x sehari</li>
                            <li><i class="fas fa-check"></i> Dokumentasi</li>
                        </ul>
                        <div class="package-price">
                            <span class="price">Rp ${pkg.price}</span>
                            <span class="duration">/ orang (${pkg.duration})</span>
                        </div>
                        <button class="btn-book-now" onclick="openModal('modal-booking')">Pesan Sekarang</button>
                    </div>
                </div>
            `;
        }
        
        return html;
    }
    
    function generateBlogPosts(count) {
        const blogs = [
            { title: 'Panduan Lengkap Mendaki Gunung di Indonesia', category: 'Tips', date: '15 April 2025', author: 'Maya Wijaya' },
            { title: '10 Destinasi Tersembunyi di Pulau Jawa', category: 'Destinasi', date: '28 Maret 2025', author: 'Adi Nugroho' },
            { title: 'Cara Memilih Peralatan Snorkeling yang Tepat', category: 'Peralatan', date: '10 Maret 2025', author: 'Reza Pratama' },
            { title: 'Mengenal Kuliner Khas Indonesia Timur', category: 'Kuliner', date: '5 Maret 2025', author: 'Diana Putri' }
        ];
        
        let html = '';
        
        for (let i = 0; i < count; i++) {
            const index = Math.floor(Math.random() * blogs.length);
            const blog = blogs[index];
            
            html += `
                <div class="blog-post">
                    <div class="blog-image">
                        <img src="/api/placeholder/400/250" alt="${blog.title}">
                        <div class="blog-category">${blog.category}</div>
                    </div>
                    <div class="blog-content">
                        <div class="blog-meta">
                            <span><i class="far fa-calendar"></i> ${blog.date}</span>
                            <span><i class="far fa-user"></i> ${blog.author}</span>
                        </div>
                        <h3>${blog.title}</h3>
                        <p>Artikel informatif yang memberikan wawasan mendalam tentang berbagai aspek perjalanan dan petualangan di Indonesia.</p>
                        <a href="#" class="read-more">Baca Selengkapnya</a>
                    </div>
                </div>
            `;
        }
        
        return html;
    }
    
    function generateGalleryItems(count) {
        const galleries = [
            { title: 'Sunset di Borobudur', category: 'budaya', description: 'Menikmati matahari terbenam di Candi Borobudur' },
            { title: 'Danau Sentani', category: 'danau', description: 'Keindahan Danau Sentani dari ketinggian' },
            { title: 'Orangutan Kalimantan', category: 'hutan', description: 'Mengamati orangutan di habitat aslinya' },
            { title: 'Terumbu Karang Wakatobi', category: 'laut', description: 'Keindahan bawah laut Wakatobi' },
            { title: 'Puncak Mahameru', category: 'gunung', description: 'Pemandangan dari puncak tertinggi Pulau Jawa' },
            { title: 'Rumah Adat Toraja', category: 'budaya', description: 'Arsitektur unik rumah adat Toraja' }
        ];
        
        let html = '';
        
        for (let i = 0; i < count; i++) {
            const index = Math.floor(Math.random() * galleries.length);
            const gallery = galleries[index];
            
            html += `
                <div class="gallery-item" data-category="${gallery.category}">
                    <img src="/api/placeholder/300/300" alt="${gallery.title}">
                    <div class="gallery-overlay">
                        <h3>${gallery.title}</h3>
                        <p>${gallery.description}</p>
                    </div>
                </div>
            `;
        }
        
        return html;
    }
    
    function initNewContentElements(container) {
        // Initialize new favorite buttons, share buttons, etc.
        const newCards = container.querySelectorAll('.destination-card:not(.initialized), .package-card:not(.initialized)');
        
        newCards.forEach(card => {
            // Add favorite button
            const favoriteButton = document.createElement('button');
            favoriteButton.className = 'favorite-button';
            favoriteButton.innerHTML = '<i class="far fa-heart"></i>';
            
            favoriteButton.addEventListener('click', function(e) {
                e.stopPropagation();
                this.classList.toggle('active');
                
                if (this.classList.contains('active')) {
                    this.innerHTML = '<i class="fas fa-heart"></i>';
                } else {
                    this.innerHTML = '<i class="far fa-heart"></i>';
                }
            });
            
            card.appendChild(favoriteButton);
            
            // Add share button
            const shareButton = document.createElement('button');
            shareButton.className = 'share-button';
            shareButton.innerHTML = '<i class="fas fa-share-alt"></i>';
            
            shareButton.addEventListener('click', function(e) {
                e.stopPropagation();
                // Share functionality would go here
            });
            
            card.appendChild(shareButton);
            
            // Mark as initialized
            card.classList.add('initialized');
        });
        
        // Update gallery filters
        if (container.classList.contains('gallery-container')) {
            const galleryItems = container.querySelectorAll('.gallery-item');
            const filterButtons = document.querySelectorAll('.filter-btn');
            
            filterButtons.forEach(button => {
                const filter = button.getAttribute('data-filter');
                
                if (filter === 'all') return;
                
                // Count items for each category
                const count = document.querySelectorAll(`.gallery-item[data-category="${filter}"]`).length;
                
                // Update count in button if it exists
                const countSpan = button.querySelector('.count');
                
                if (countSpan) {
                    countSpan.textContent = count;
                } else {
                    button.innerHTML += `<span class="count">${count}</span>`;
                }
            });
        }
    }
}

function initDynamicBanners() {
    // Tambahkan banner promosi yang dinamis
    const banners = [
        {
            title: 'Diskon 20% untuk Pendakian Gunung',
            description: 'Pesan paket pendakian sebelum akhir bulan dan dapatkan diskon 20%',
            background: 'linear-gradient(135deg, #43cea2 0%, #185a9d 100%)',
            icon: 'fas fa-mountain',
            buttonText: 'Lihat Penawaran',
            buttonAction: 'openModal("modal-paket1")'
        },
        {
            title: 'Kuota Terbatas: Trip Raja Ampat',
            description: 'Hanya tersisa 5 slot untuk trip Raja Ampat bulan depan',
            background: 'linear-gradient(135deg, #ff7e5f 0%, #feb47b 100%)',
            icon: 'fas fa-water',
            buttonText: 'Pesan Sekarang',
            buttonAction: 'openModal("modal-booking")'
        },
        {
            title: 'Paket Keluarga Liburan Sekolah',
            description: 'Nikmati liburan bersama keluarga dengan paket spesial liburan sekolah',
            background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
            icon: 'fas fa-users',
            buttonText: 'Lihat Paket',
            buttonAction: 'scrollToSection("paket")'
        }
    ];
    
    // Pilih banner secara acak
    const randomBanner = banners[Math.floor(Math.random() * banners.length)];
    
    // Buat banner element
    const banner = document.createElement('div');
    banner.className = 'promo-banner';
    banner.style.background = randomBanner.background;
    
    banner.innerHTML = `
        <div class="banner-icon">
            <i class="${randomBanner.icon}"></i>
        </div>
        <div class="banner-content">
            <h3>${randomBanner.title}</h3>
            <p>${randomBanner.description}</p>
        </div>
        <button class="banner-button" onclick="${randomBanner.buttonAction}">${randomBanner.buttonText}</button>
        <button class="close-banner"><i class="fas fa-times"></i></button>
    `;
    
    // Tambahkan ke halaman
    const heroSection = document.querySelector('.hero-section');
    
    if (heroSection) {
        document.body.insertBefore(banner, heroSection.nextSibling);
    } else {
        document.body.appendChild(banner);
    }
    
    // Tambahkan event listener untuk tombol close
    const closeButton = banner.querySelector('.close-banner');
    
    if (closeButton) {
        closeButton.addEventListener('click', function() {
            banner.classList.add('closing');
            
            setTimeout(() => {
                banner.remove();
            }, 300);
        });
    }
    
    // Function untuk scroll ke section tertentu
    window.scrollToSection = function(sectionId) {
        const section = document.getElementById(sectionId);
        
        if (section) {
            window.scrollTo({
                top: section.offsetTop - 80,
                behavior: 'smooth'
            });
        }
    };
}

function initSocialProof() {
    // Tambahkan social proof notification
    const notifications = [
        { name: 'Andi dari Jakarta', action: 'memesan', item: 'Paket Pendakian Rinjani', time: '5 menit yang lalu' },
        { name: 'Ratna dari Surabaya', action: 'memesan', item: 'Paket Diving Raja Ampat', time: '12 menit yang lalu' },
        { name: 'Budi dari Bandung', action: 'melihat', item: 'Paket Eksplorasi Komodo', time: 'baru saja' },
        { name: 'Sinta dari Medan', action: 'menambahkan ke favorit', item: 'Gunung Bromo', time: '3 menit yang lalu' },
        { name: '27 orang lainnya', action: 'melihat', item: 'Paket Petualangan Populer', time: 'hari ini' }
    ];
    
    let currentNotification = 0;
    
    // Buat notification element
    const notification = document.createElement('div');
    notification.className = 'social-proof';
    
    updateNotification();
    
    // Tambahkan ke halaman
    document.body.appendChild(notification);
    
    // Tampilkan notification pertama setelah 10 detik
    setTimeout(() => {
        showNotification();
    }, 10000);
    
    function updateNotification() {
        const data = notifications[currentNotification];
        
        notification.innerHTML = `
            <div class="social-proof-icon">
                <i class="fas fa-${data.action === 'memesan' ? 'shopping-cart' : data.action === 'melihat' ? 'eye' : 'heart'}"></i>
            </div>
            <div class="social-proof-content">
                <p><strong>${data.name}</strong> ${data.action} <strong>${data.item}</strong></p>
                <span>${data.time}</span>
            </div>
            <button class="close-social-proof"><i class="fas fa-times"></i></button>
        `;
        
        // Event listener untuk tombol close
        const closeButton = notification.querySelector('.close-social-proof');
        
        if (closeButton) {
            closeButton.addEventListener('click', function() {
                hideNotification();
            });
        }
    }
    
    function showNotification() {
        notification.classList.add('show');
        
        // Sembunyikan setelah 5 detik
        setTimeout(() => {
            hideNotification();
        }, 5000);
    }
    
    function hideNotification() {
        notification.classList.remove('show');
        
        // Update ke notification berikutnya
        currentNotification = (currentNotification + 1) % notifications.length;
        
        setTimeout(() => {
            updateNotification();
            
            // Tampilkan notification berikutnya setelah interval acak
            const delay = Math.floor(Math.random() * 20000) + 15000; // 15-35 detik
            
            setTimeout(() => {
                showNotification();
            }, delay);
        }, 500);
    }
}

function initCustomCursor() {
    // Buat kursor kustom untuk meningkatkan pengalaman visual
    const cursor = document.createElement('div');
    cursor.className = 'custom-cursor';
    
    const cursorDot = document.createElement('div');
    cursorDot.className = 'cursor-dot';
    
    const cursorRing = document.createElement('div');
    cursorRing.className = 'cursor-ring';
    
    cursor.appendChild(cursorDot);
    cursor.appendChild(cursorRing);
    
    document.body.appendChild(cursor);
    
    // Update posisi kursor
    document.addEventListener('mousemove', (e) => {
        cursor.style.display = 'block';
        cursorDot.style.transform = `translate(${e.clientX}px, ${e.clientY}px)`;
        cursorRing.style.transform = `translate(${e.clientX}px, ${e.clientY}px)`;
    });
    
    // Sembunyikan kursor saat mouse keluar window
    document.addEventListener('mouseout', () => {
        cursor.style.display = 'none';
    });
    
    // Tambahkan efek hover pada elemen yang bisa di-klik
    const clickables = document.querySelectorAll('a, button, .btn, .destination-card, .package-card, .gallery-item');
    
    clickables.forEach(element => {
        element.addEventListener('mouseover', () => {
            cursorRing.classList.add('hover');
        });
        
        element.addEventListener('mouseout', () => {
            cursorRing.classList.remove('hover');
        });
        
        element.addEventListener('mousedown', () => {
            cursorRing.classList.add('click');
        });
        
        element.addEventListener('mouseup', () => {
            cursorRing.classList.remove('click');
        });
    });
    
  // Deteksi perangkat mobile
  function isMobileDevice() {
    return (typeof window.orientation !== "undefined") 
        || (navigator.userAgent.indexOf('IEMobile') !== -1)
        || (window.innerWidth <= 768);
}

// Nonaktifkan kursor kustom pada perangkat mobile
if (isMobileDevice()) {
    cursor.style.display = 'none';
}
}

function initVirtualTour() {
// Tambahkan virtual tour untuk destinasi populer
const destinations = document.querySelectorAll('.destination-card');

destinations.forEach(destination => {
    const title = destination.querySelector('h3').textContent;
    
    // Tambahkan tombol virtual tour
    const tourButton = document.createElement('button');
    tourButton.className = 'virtual-tour-button';
    tourButton.innerHTML = '<i class="fas fa-vr-cardboard"></i> Virtual Tour';
    
    tourButton.addEventListener('click', function(e) {
        e.stopPropagation();
        openVirtualTour(title);
    });
    
    const btnContainer = destination.querySelector('.destination-info');
    
    if (btnContainer) {
        const viewButton = btnContainer.querySelector('.btn-view-details');
        
        if (viewButton) {
            btnContainer.insertBefore(tourButton, viewButton);
        } else {
            btnContainer.appendChild(tourButton);
        }
    }
});

function openVirtualTour(destinationName) {
    // Buat modal untuk virtual tour
    const virtualTourModal = document.createElement('div');
    virtualTourModal.className = 'modal';
    virtualTourModal.id = 'modal-virtual-tour';
    
    virtualTourModal.innerHTML = `
        <div class="modal-content fullscreen-modal">
            <span class="close-modal" onclick="closeModal('modal-virtual-tour')">&times;</span>
            <div class="virtual-tour-container">
                <div class="tour-loading">
                    <i class="fas fa-spinner fa-spin"></i>
                    <p>Memuat Virtual Tour...</p>
                </div>
                <div class="tour-controls">
                    <button class="tour-control" data-direction="left"><i class="fas fa-arrow-left"></i></button>
                    <button class="tour-control" data-direction="up"><i class="fas fa-arrow-up"></i></button>
                    <button class="tour-control" data-direction="down"><i class="fas fa-arrow-down"></i></button>
                    <button class="tour-control" data-direction="right"><i class="fas fa-arrow-right"></i></button>
                    <button class="tour-control zoom-in"><i class="fas fa-search-plus"></i></button>
                    <button class="tour-control zoom-out"><i class="fas fa-search-minus"></i></button>
                </div>
                <div class="tour-info">
                    <h2>Virtual Tour ${destinationName}</h2>
                    <p>Gunakan panah atau geser untuk menjelajahi area.</p>
                </div>
                <div class="tour-view">
                    <img src="/img/rinjani-modal/1.jpg" alt="Virtual Tour ${destinationName}" class="tour-image">
                </div>
                <div class="tour-thumbnail-strip">
                    <div class="tour-thumbnail active" data-scene="scene1">
                        <img src="/img/rinjani-modal/1.jpg" alt="Thumbnail 1">
                    </div>
                    <div class="tour-thumbnail" data-scene="scene2">
                        <img src="/img/rinjani-modal/2.jpg" alt="Thumbnail 2">
                    </div>
                    <div class="tour-thumbnail" data-scene="scene3">
                        <img src="/img/rinjani-modal/3.jpg" alt="Thumbnail 3">
                    </div>
                    <div class="tour-thumbnail" data-scene="scene4">
                        <img src="/img/rinjani-modal/4.jpg" alt="Thumbnail 4">
                    </div>
                    <div class="tour-thumbnail" data-scene="scene5">
                        <img src="/img/rinjani-modal/4.jpg" alt="Thumbnail 5">
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(virtualTourModal);
    
    // Buka modal
    openModal('modal-virtual-tour');
    
    // Simulasi loading
    setTimeout(() => {
        const loadingElement = virtualTourModal.querySelector('.tour-loading');
        if (loadingElement) {
            loadingElement.style.display = 'none';
        }
        
        // Inisialisasi virtual tour
        initTourControls(virtualTourModal);
    }, 2000);
}

function initTourControls(modal) {
    // Thumbnail navigation
    const thumbnails = modal.querySelectorAll('.tour-thumbnail');
    const tourImage = modal.querySelector('.tour-image');
    
    thumbnails.forEach(thumbnail => {
        thumbnail.addEventListener('click', function() {
            // Deactivate all thumbnails
            thumbnails.forEach(thumb => thumb.classList.remove('active'));
            
            // Activate clicked thumbnail
            this.classList.add('active');
            
            // Update main image (in a real app, this would load a new panorama)
            const scene = this.getAttribute('data-scene');
            tourImage.classList.add('transition');
            
            setTimeout(() => {
                tourImage.src = `/api/placeholder/1200/800?scene=${scene}`;
                tourImage.classList.remove('transition');
            }, 300);
        });
    });
    
    // Direction controls
    const controls = modal.querySelectorAll('.tour-control');
    
    controls.forEach(control => {
        control.addEventListener('click', function() {
            const direction = this.getAttribute('data-direction');
            
            if (direction) {
                // Simulate movement in that direction
                tourImage.classList.add(`move-${direction}`);
                
                setTimeout(() => {
                    tourImage.classList.remove(`move-${direction}`);
                }, 300);
            } else if (this.classList.contains('zoom-in')) {
                // Zoom in
                tourImage.classList.add('zoom-in-effect');
                
                setTimeout(() => {
                    tourImage.classList.remove('zoom-in-effect');
                }, 300);
            } else if (this.classList.contains('zoom-out')) {
                // Zoom out
                tourImage.classList.add('zoom-out-effect');
                
                setTimeout(() => {
                    tourImage.classList.remove('zoom-out-effect');
                }, 300);
            }
        });
    });
    
    // Mouse drag panning
    let isDragging = false;
    let prevX = 0;
    let prevY = 0;
    
    tourImage.addEventListener('mousedown', function(e) {
        isDragging = true;
        prevX = e.clientX;
        prevY = e.clientY;
        tourImage.classList.add('dragging');
    });
    
    document.addEventListener('mousemove', function(e) {
        if (!isDragging) return;
        
        const deltaX = e.clientX - prevX;
        const deltaY = e.clientY - prevY;
        
        // Simulate panning based on drag direction
        if (Math.abs(deltaX) > Math.abs(deltaY)) {
            if (deltaX > 0) {
                tourImage.classList.add('pan-right');
            } else {
                tourImage.classList.add('pan-left');
            }
        } else {
            if (deltaY > 0) {
                tourImage.classList.add('pan-down');
            } else {
                tourImage.classList.add('pan-up');
            }
        }
        
        prevX = e.clientX;
        prevY = e.clientY;
    });
    
    document.addEventListener('mouseup', function() {
        if (!isDragging) return;
        
        isDragging = false;
        tourImage.classList.remove('dragging', 'pan-left', 'pan-right', 'pan-up', 'pan-down');
    });
}
}

function initWebSocketDemo() {
// Simulasi koneksi WebSocket untuk fitur-fitur real-time
console.log("Initializing WebSocket demo features...");

// Simulasi activity feed
const activities = [
    { user: "Andi", action: "baru saja memesan Paket Pendakian Rinjani" },
    { user: "Maya", action: "menambahkan Raja Ampat ke wishlist" },
    { user: "Budi", action: "memberikan rating 5 bintang untuk trip Bromo" },
    { user: "Sinta", action: "baru saja bergabung dengan Camp IDN" },
    { user: "Reza", action: "membagikan pengalaman di Taman Nasional Komodo" }
];

// Simulasi chat support
initChatSupport();

// Simulasi notifikasi seat availability
simulateSeatAvailability();

// Fungsi-fungsi simulasi
function initChatSupport() {
    const chatButton = document.createElement('div');
    chatButton.className = 'chat-support-button';
    chatButton.innerHTML = '<i class="fas fa-comment-dots"></i>';
    
    chatButton.addEventListener('click', function() {
        toggleChatWindow();
    });
    
    document.body.appendChild(chatButton);
    
    const chatWindow = document.createElement('div');
    chatWindow.className = 'chat-window';
    chatWindow.innerHTML = `
        <div class="chat-header">
            <div class="chat-agent">
                <img src="/img/chat/live chat.png" alt="Agent">
                <div class="agent-info">
                    <h4>Tim Support</h4>
                    <span>Online</span>
                </div>
            </div>
            <button class="close-chat"><i class="fas fa-times"></i></button>
        </div>
        <div class="chat-messages">
            <div class="message-system">Selamat datang di Camp IDN Live Chat Support</div>
            <div class="message-agent">
                <div class="message-avatar">
                    <img src="/img/chat/live chat.png" alt="Agent">
                </div>
                <div class="message-content">
                    <span class="message-name">Rina</span>
                    <div class="message-bubble">
                        Halo! Terima kasih telah menghubungi Camp IDN. Ada yang bisa saya bantu hari ini?
                    </div>
                    <span class="message-time">11:24</span>
                </div>
            </div>
        </div>
        <div class="chat-input">
            <input type="text" placeholder="Ketik pesan Anda..." />
            <button class="send-message"><i class="fas fa-paper-plane"></i></button>
        </div>
    `;
    
    document.body.appendChild(chatWindow);
    
    const closeButton = chatWindow.querySelector('.close-chat');
    const inputField = chatWindow.querySelector('.chat-input input');
    const sendButton = chatWindow.querySelector('.send-message');
    const messagesContainer = chatWindow.querySelector('.chat-messages');
    
    closeButton.addEventListener('click', function() {
        toggleChatWindow();
    });
    
    sendButton.addEventListener('click', function() {
        sendMessage();
    });
    
    inputField.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            sendMessage();
        }
    });
    
    function toggleChatWindow() {
        chatWindow.classList.toggle('open');
        chatButton.classList.toggle('hidden');
        
        if (chatWindow.classList.contains('open')) {
            inputField.focus();
        }
    }
    
    function sendMessage() {
        const message = inputField.value.trim();
        
        if (!message) return;
        
        // Add user message
        const now = new Date();
        const timeString = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;
        
        const userMessageHTML = `
            <div class="message-user">
                <div class="message-content">
                    <div class="message-bubble">
                        ${message}
                    </div>
                    <span class="message-time">${timeString}</span>
                </div>
            </div>
        `;
        
        messagesContainer.insertAdjacentHTML('beforeend', userMessageHTML);
        
        // Clear input
        inputField.value = '';
        
        // Scroll to bottom
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        
        // Simulate typing
        const typingHTML = `
            <div class="message-typing">
                <div class="message-avatar">
                    <img src="/img/chat/live chat.png" alt="Agent">
                </div>
                <div class="typing-indicator">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
            </div>
        `;
        
        messagesContainer.insertAdjacentHTML('beforeend', typingHTML);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        
        // Simulate response after delay
        setTimeout(() => {
            // Remove typing indicator
            const typingIndicator = messagesContainer.querySelector('.message-typing');
            if (typingIndicator) {
                typingIndicator.remove();
            }
            
            // Add agent response
            const responseTime = new Date();
            const responseTimeString = `${responseTime.getHours().toString().padStart(2, '0')}:${responseTime.getMinutes().toString().padStart(2, '0')}`;
            
            const responses = [
                "Terima kasih atas pertanyaan Anda. Bisa saya bantu memberikan informasi lebih lanjut?",
                "Baik, saya mengerti pertanyaan Anda. Apakah ada informasi spesifik yang Anda butuhkan?",
                "Untuk informasi tersebut, saya sarankan Anda melihat paket perjalanan kami di bagian Paket Petualangan.",
                "Ya, kami memiliki trip ke destinasi tersebut. Apakah Anda ingin saya kirimkan detail paketnya?",
                "Untuk pemesanan, Anda bisa klik tombol 'Pesan Sekarang' pada paket yang Anda minati."
            ];
            
            const randomResponse = responses[Math.floor(Math.random() * responses.length)];
            
            const agentMessageHTML = `
                <div class="message-agent">
                    <div class="message-avatar">
                        <img src="/img/chat/live chat.png" alt="Agent">
                    </div>
                    <div class="message-content">
                        <span class="message-name">Rina</span>
                        <div class="message-bubble">
                            ${randomResponse}
                        </div>
                        <span class="message-time">${responseTimeString}</span>
                    </div>
                </div>
            `;
            
            messagesContainer.insertAdjacentHTML('beforeend', agentMessageHTML);
            
            // Scroll to bottom
            messagesContainer.scrollTop = messagesContainer.scrollHeight;
        }, 1500);
    }
}

function simulateSeatAvailability() {
    // Tambahkan indikator ketersediaan kursi pada paket perjalanan
    const packageCards = document.querySelectorAll('.package-card');
    
    packageCards.forEach(card => {
        const randomAvailability = Math.floor(Math.random() * 10) + 1;
        const isLimited = randomAvailability <= 3;
        
        const availabilityBadge = document.createElement('div');
        availabilityBadge.className = `seat-availability ${isLimited ? 'limited' : ''}`;
        
        availabilityBadge.innerHTML = `
            <i class="fas fa-users"></i>
            <span>${isLimited ? 'Sisa ' + randomAvailability + ' slot' : 'Tersedia'}</span>
        `;
        
        card.appendChild(availabilityBadge);
        
        if (isLimited) {
            // Simulate live updates for limited seats
            setInterval(() => {
                const currentAvailability = parseInt(availabilityBadge.querySelector('span').textContent.match(/\d+/)[0]);
                
                if (Math.random() > 0.7 && currentAvailability > 1) {
                    const newAvailability = currentAvailability - 1;
                    availabilityBadge.querySelector('span').textContent = `Sisa ${newAvailability} slot`;
                    
                    // Show notification
                    showNotification('Slot Berkurang', `Seseorang baru saja memesan paket ini. Sisa ${newAvailability} slot tersedia.`, 'warning');
                }
            }, 30000); // Check every 30 seconds
        }
    });
}
}

function initPageExitIntent() {
// Deteksi exit intent untuk menampilkan popup
let exitShown = false;

document.addEventListener('mouseleave', function(e) {
    if (e.clientY < 0 && !exitShown) {
        showExitPopup();
        exitShown = true;
    }
});

function showExitPopup() {
    // Cek apakah user sudah berlangganan newsletter
    const hasSubscribed = localStorage.getItem('campidn_newsletter');
    
    if (hasSubscribed) return;
    
    // Buat exit popup
    const exitPopup = document.createElement('div');
    exitPopup.className = 'modal exit-popup';
    exitPopup.id = 'modal-exit';
    
    exitPopup.innerHTML = `
        <div class="modal-content">
            <span class="close-modal" onclick="closeModal('modal-exit')">&times;</span>
            <div class="exit-popup-container">
                <div class="exit-popup-image">
                    <img src="/img/img-paket-diskon/paket.png" alt="Special Offer">
                </div>
                <div class="exit-popup-content">
                    <h2>Tunggu Sebentar!</h2>
                    <h3>Dapatkan Diskon 10% untuk Pemesanan Pertama</h3>
                    <p>Berlangganan newsletter kami dan dapatkan kode promo eksklusif langsung ke email Anda.</p>
                    
                    <form class="exit-form">
                        <div class="form-group">
                            <input type="email" placeholder="Alamat Email Anda" required>
                        </div>
                        <button type="submit" class="btn-subscribe">Berlangganan & Dapatkan Diskon</button>
                    </form>
                    
                    <div class="form-footer">
                        <p>Kami tidak akan pernah membagikan email Anda dengan pihak lain.</p>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(exitPopup);
    
    // Tambahkan event listener untuk form
    const exitForm = exitPopup.querySelector('.exit-form');
    
    if (exitForm) {
        exitForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const emailInput = this.querySelector('input[type="email"]');
            const email = emailInput.value.trim();
            
            if (!email || !validateEmail(email)) {
                emailInput.classList.add('invalid');
                return;
            }
            
            // Show loading state
            const submitButton = this.querySelector('button[type="submit"]');
            submitButton.disabled = true;
            submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Memproses...';
            
            // Simulate subscription process
            setTimeout(() => {
                localStorage.setItem('campidn_newsletter', email);
                
                // Show success message
                const formContainer = exitForm.parentElement;
                
                formContainer.innerHTML = `
                    <div class="subscription-success">
                        <i class="fas fa-check-circle"></i>
                        <h3>Terima Kasih Telah Berlangganan!</h3>
                        <p>Kode promo Anda: <strong>WELCOME10</strong></p>
                        <p>Kode promo telah dikirim ke email Anda dan dapat digunakan untuk mendapatkan diskon 10% pada pemesanan pertama.</p>
                        <button class="btn-primary" onclick="closeModal('modal-exit')">Lanjutkan Menjelajah</button>
                    </div>
                `;
            }, 2000);
        });
    }
    
    openModal('modal-exit');
}

function validateEmail(email) {
    const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return regex.test(email);
}
}

// Eksekusi fungsi-fungsi tambahan
window.addEventListener('load', function() {
// Inisialisasi fitur dasar sudah ada di DOMContentLoaded
// Inisialisasi fitur-fitur tambahan (lanjutan)
setTimeout(() => {
    initImageGallery();
    initAdvancedFilters();
    initProgressiveLoading();
    initDynamicBanners();
    initSocialProof();
    initCustomCursor();
    initVirtualTour();
    initWebSocketDemo();
    initPageExitIntent();
}, 1000); // Delay untuk memastikan konten dasar sudah dimuat
});

// Eksport fungsi global yang digunakan dalam HTML
window.openModal = openModal;
window.closeModal = closeModal;
window.toggleMenu = toggleMenu;
window.changeMainImage = changeMainImage;
window.closeNotification = closeNotification;